# -*- coding: utf-8 -*-
"""
Created on Fri Aug  4 13:42:31 2017

@author: Sophie

"""

import ctypes
import os
from enum import Enum
from enum import Flag
import weakref
import platform

if platform.architecture()[0] == '64bit':
    Netica = ctypes.WinDLL(os.path.abspath('../bin/Netica.dll'))
elif platform.architecture()[0] == '32bit':
    Netica = ctypes.WinDLL(os.path.abspath('../bin/32bit/Netica.dll'))
else:
    raise OSError("Can't find the dll file Netica.dll, which should be contained \
                  in /bin for 64 bit python or /bin/32bit for 32 bit python.") 

# Choose whether to identify objects using dict or userdata field
# Only one can be true at a time

# Some features may not work with dict_initialization = False (multiple python 
# objects may be created for each Netica object)
dict_initialization = True
userdata_initialization = False

# Enumeration

QUERY_ns = -1

class ErrorSeverity(Enum):
    NOTHING = 1
    REPORT = 2
    NOTICE = 3
    WARNING = 4
    ERROR = 5
    XXX = 6

class ErrorCondition(Enum):
    OUT_OF_MEMORY_CND = 0x08 
    USER_ABORTED_CND = 0x20
    FROM_WRAPPER_CND = 0x40
    FROM_DEVELOPER_CND = 0x80
    INCONS_FINDING_CND = 0x200

class NodeType(Enum):
    CONTINUOUS = 1
    DISCRETE = 2
    TEXT = 3

class NodeKind(Enum):
    NATURE = 1
    CONSTANT = 2
    DECISION = 3
    UTILITY = 4
    DISCONNECTED = 5
    ADVERSARY = 6

class ReadingOption(Flag):
    NO_VISUAL_INFO = 0
    NO_WINDOW = 0x10
    MINIMIZED_WINDOW = 0x30
    REGULAR_WINDOW = 0x70
    
class LearningMethod(Enum):
    COUNTING = 1 
    EM = 3
    GRADIENT_DESCENT = 4

class SamplingMethod(Enum):
    DEFAULT = 0
    JOIN_TREE = 1
    FORWARD = 2

class Sensv(Flag):
    ENTROPY = 0x02
    REAL = 0x04
    VARIANCE = 0x100 
    VARIANCE_OF_REAL = 0x104

class CasePosition(Enum):
    FIRST_CASE = -15
    NEXT_CASE = -14
    NO_MORE_CASES = -13

class ArgumentChecking(Enum):
    NO_CHECK = 1
    QUICK_CHECK = 2
    REGULAR_CHECK = 3
    COMPLETE_CHECK = 4
    QUERY_CHECK = -1

class State(Enum):
    EVERY_STATE = -5
    IMPOSS_STATE = -4
    UNDEF_STATE = -3

class Finding(Enum):
    NEGATIVE_FINDING = -7
    LIKELIHOOD_FINDING = -6
    NO_FINDING = -3

class NeticaPyError(Exception):
    pass

# Error numbers 6500 - 6599 

def checkerr():
    if checkerr:
        Netica.GetError_ns.restype = ctypes.c_void_p
        report = Netica.GetError_ns(ctypes.c_void_p(env), 5, None)
        if report:
            Netica.ErrorMessage_ns.restype = ctypes.c_char_p
            net_errmesg = Netica.ErrorMessage_ns(ctypes.c_void_p(report))
            errmesg = str('Netica Error ' + str(Netica.ErrorNumber_ns(ctypes.c_void_p(report))) +  ': ' + net_errmesg.decode())         
            Netica.ClearErrors_ns.restype = None
            Netica.ClearErrors_ns(ctypes.c_void_p(env), 6) # May not want to clear XXX_ERR to force the 
            #developer to restart the system when encountereing an internal error (from 6 to 5)
            #Netica.ClearError_ns.restype = None
            #Netica.ClearError_ns(ctypes.c_void_p(report))
            raise NeticaPyError(errmesg)

# Initialize dict that will contain the c pointers for all objects
# Search when initializing a new object to return an existing python object

if dict_initialization:
    cptr_dict = {}
    
CALLBACK = ctypes.WINFUNCTYPE(ctypes.c_int, ctypes.c_void_p, ctypes.c_int,
                              ctypes.c_void_p, ctypes.c_void_p)

@CALLBACK
def net_callback(net, event, obj, data):
    if event == 4:
        if dict_initialization:
            if net in cptr_dict:
                py_net = cptr_dict[net]()
                del cptr_dict[py_net.cptr]
                py_net.cptr = None
    return 0

@CALLBACK
def node_callback(node, event, obj, data):
    if event == 4:
        if dict_initialization: 
            if node in cptr_dict:
                py_node = cptr_dict[node]()   
                del cptr_dict[py_node.cptr]                     
                py_node.cptr = None
    return 0


def create_callback(function):
    callback = ctypes.WINFUNCTYPE(ctypes.c_int, ctypes.c_void_p, ctypes.c_int,
                                      ctypes.c_void_p, ctypes.c_void_p)
    return callback(function)


def create_net(cptr):
    """Function called by NeticaPy to create a net.
    
    cptr is the C pointer generated by a call to a Netica DLL function. 
    create_net uses the pointer to check whether a NeticaPy instance of this 
    net already exists, and returns any existing net. A new instance will be 
    created if the pointer isn't present in cptr_dict.
    """
    if dict_initialization:
        if cptr in cptr_dict:
            return cptr_dict[cptr]()
        else:
            return Net(cptr)
    
    elif userdata_initialization:
        existing_net = Netica.GetNetUserData_bn(ctypes.c_void_p(cptr), 0)
        checkerr()
        if existing_net:
            py_net = ctypes.cast(existing_net, 
                                 ctypes.POINTER(ctypes.py_object)).contents.value
            return py_net
        else:
            return Net(cptr)
    else:
        return Net(cptr)

def create_node(cptr):
    """Function called by NeticaPy to create a node.
    
    cptr is the C pointer generated by a call to a Netica DLL function. 
    create_node uses the pointer to check whether a NeticaPy instance of this 
    node already exists, and returns any existing node. A new instance will be 
    created if the pointer isn't present in cptr_dict.
    """
    if dict_initialization:
        if cptr in cptr_dict:
            return cptr_dict[cptr]()
        else:
            return Node(cptr)

    elif userdata_initialization:
        existing_node = Netica.GetNodeUserData_bn(ctypes.c_void_p(cptr), 0)
        checkerr()
        if existing_node:
            py_node = ctypes.cast(existing_node, 
                                 ctypes.POINTER(ctypes.py_object)).contents.value
            return py_node
        else:
            return Node(cptr)   
    else:
        return Node(cptr)
   
def create_nodelist(cptr, is_const=None):
    """Function called by NeticaPy to create a nodelist.
    
    cptr is the C pointer generated by a call to a Netica DLL function. 
    create_nodelist uses the pointer to check whether a NeticaPy instance of 
    this nodelist already exists, and returns any existing nodelist. A new 
    instance will be created if the pointer isn't present in cptr_dict.
    """
    if dict_initialization:
        if cptr in cptr_dict:
            return cptr_dict[cptr]()
        else:
            return NodeList(cptr, is_const)
    else:
        return NodeList(cptr, is_const)
    
"""------------------------------Environ Class------------------------------"""
   
class Environ:
    
    def __init__(self, license_=None, environ=None, locn=None):
        
        if license_ is not None:
            license_ = ctypes.c_char_p(license_.encode())
        if locn is not None:
            locn = ctypes.c_char_p(locn.encode())
            
        Netica.NewNeticaEnviron_ns.restype = ctypes.c_void_p
        global env 
        env = Netica.NewNeticaEnviron_ns(license_, environ, locn)
        self.cptr = env
    
    def init_netica(self, mesg=None):         
        """Initialize Netica Environment
        
        This functioncalls the Netica-C functions InitNetica2_bn, SetLanguage_ns,
        AddNetListener_bn, AddNodeListener_bn, GetUndefDbl_ns, and 
        GetInfinityDbl_ns to set up a Netica environment through Python.
        """
        mesg = ctypes.c_char_p(b'')
                
        Netica.InitNetica2_bn.restype = ctypes.c_int
        res, mesg = Netica.InitNetica2_bn(ctypes.c_void_p(self.cptr), mesg), mesg.value.decode()
        checkerr()
        
        Netica.SetLanguage_ns.restype = ctypes.c_char_p
        Netica.SetLanguage_ns(ctypes.c_void_p(self.cptr), 
                              ctypes.c_char_p("Python".encode()))
        checkerr()
        
        # Set up global listeners for callback deletion
        
        Netica.AddNetListener_bn.restype = None
        Netica.AddNetListener_bn(None, net_callback, None, -1)
        
        Netica.AddNodeListener_bn.restype = None
        Netica.AddNodeListener_bn(None, node_callback, None, -1)
    
        global UNDEF_DBL
        Netica.GetUndefDbl_ns.restype = ctypes.c_double
        UNDEF_DBL = Netica.GetUndefDbl_ns()
        
        global INFINITY
        Netica.GetInfinityDbl_ns.retype = ctypes.c_double
        INFINITY = Netica.GetInfinityDbl_ns()

        checkerr()
    
        return res, mesg
        
    def close_netica(self, mesg=None):
        """Exit Netica (i.e. make it quit).  
        
        If a human user has entered unsaved changes, this will check with the user first.  
        Like Netica-C CloseNetica_bn.
        """
        mesg = ctypes.c_char_p(b'')
        
        Netica.CloseNetica_bn.restype = ctypes.c_int
        return Netica.CloseNetica_bn(ctypes.c_void_p(self.cptr), mesg), mesg.value.decode()
    
    @property
    def version_num(self):
        """Return version number of Netica.
        
        Version is multiplied by 100, so version 3.24 would return 324.  
        Like Netica-C GetNeticaVersion_bn.
        """
        Netica.GetNeticaVersion_bn.restype = ctypes.c_int
        version_num = Netica.GetNeticaVersion_bn(ctypes.c_void_p(self.cptr), None) 
        checkerr()
        return version_num
    
    @property
    def version(self):
        """Return Netica version information.
        
        This consists of the full version number, a space, a code for the type 
        of machine it is running on, a comma, the name of the program, and 
        finally a code indicating some build information.
        Like Netica-C GetNeticaVersion_bn.
        """
        version = ctypes.c_char_p(b'')
        Netica.GetNeticaVersion_bn.restype = ctypes.c_int
        Netica.GetNeticaVersion_bn(ctypes.c_void_p(self.cptr), 
                                   ctypes.byref(version)) 
        checkerr()
        return version.value.decode()
    
    @property
    def argument_checking(self):
        """To what degree Netica functions check their arguments.  
        
        Like Netica-C ArgumentChecking_ns.
        """
        Netica.ArgumentChecking_ns.restype = ctypes.c_int
        setting = Netica.ArgumentChecking_ns(ctypes.c_int(QUERY_ns), 
                                             ctypes.c_void_p(self.cptr))
        checkerr()
        return ArgumentChecking(setting).name
    
    @argument_checking.setter
    def argument_checking(self, setting):
        """To what degree Netica functions check their arguments.  
        
        Like Netica-C ArgumentChecking_ns.
        """
        Netica.ArgumentChecking_ns.restype = ctypes.c_int
        Netica.ArgumentChecking_ns(ctypes.c_int(setting.value), 
                                   ctypes.c_void_p(self.cptr)) 
        checkerr()    
    
    '''
    def set_password(self, password, options=None):
        """Sets Netica user password just for this session only.  
        
        Like the password passed to Netica-C NewNeticaEnviron_ns.
        Like Netica-C SetPassword_ns.
        Will be implemented in next API
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.SetPassword_ns.restype = None
        Netica.SetPassword_ns(ctypes.c_void_p(self.cptr), 
                              ctypes.c_char_p(password.encode()), options)
        checkerr()
        '''
    
    @property
    def memory_usage_limit(self):
        """The maximum amount of memory Netica is allowed to use (in bytes).
        
        This includes all tables and lists.  
        When setting, max_mem is the number of bytes allowed. For example, to 
        limit memory usage to 100 Megabytes, pass 100e6.
        Like Netica-C LimitMemoryUsage_ns.
        """
        Netica.LimitMemoryUsage_ns.restype = ctypes.c_double
        mem_usage_limit = Netica.LimitMemoryUsage_ns(ctypes.c_double(QUERY_ns), 
                                                     ctypes.c_void_p(self.cptr))
        checkerr()
        return mem_usage_limit
    
    @memory_usage_limit.setter
    def memory_usage_limit(self, max_mem):
        """The maximum amount of memory Netica is allowed to use (in bytes).
        
        This includes all tables and lists.  
        When setting, max_mem is the number of bytes allowed. For example, to 
        limit memory usage to 100 Megabytes, pass 100e6.
        Like Netica-C LimitMemoryUsage_ns.
        """
        Netica.LimitMemoryUsage_ns.restype = ctypes.c_double
        Netica.LimitMemoryUsage_ns(ctypes.c_double(max_mem), ctypes.c_void_p(self.cptr))
        checkerr()
        
    def test_fault_recovery(self, test_num):
        """Test Netica's fault recovery under extreme conditions.  
        
        Like Netica-C TestFaultRecovery_ns.
        """
        Netica.TestFaultRecovery_ns.restype = ctypes.c_int
        ret = Netica.TestFaultRecovery_ns(ctypes.c_void_p(self.cptr), ctypes.c_int(test_num))
        checkerr()
        return ret

    @property
    def case_file_delimiter(self):
        """The character to use as a delimeter when creating case files.  
        
        For newchar, pass the ascii character code. 
        It must be one of tab (9), space (32) or comma (44).
        Like Netica-C SetCaseFileDelimChar_ns.
        """
        Netica.SetCaseFileDelimChar_ns.restype = ctypes.c_int
        delimiter = Netica.SetCaseFileDelimChar_ns(ctypes.c_int(QUERY_ns),
                                                   ctypes.c_void_p(self.cptr))
        checkerr()
        return chr(delimiter)
        
    @case_file_delimiter.setter
    def case_file_delimiter(self, newchar):
        """The character to use as a delimeter when creating case files.
        
        For newchar, pass the ascii character code. 
        It must be one of tab (9), space (32) or comma (44).
        Like Netica-C SetCaseFileDelimChar_ns.
        """
        if isinstance(newchar, str):
            newchar = ord(newchar)
        Netica.SetCaseFileDelimChar_ns.restype = ctypes.c_int
        Netica.SetCaseFileDelimChar_ns(ctypes.c_int(newchar), ctypes.c_void_p(self.cptr))
        checkerr()
    
    @property
    def case_file_missing_data_char(self):
        """The character to use to indicate missing data when creating case files.
        
        It must be one of asterisk * (42), question mark ? (63), space (32) or 
        absent (0).  
        Like Netica-C SetMissingDataChar_ns.
        """
        Netica.SetMissingDataChar_ns.restype = ctypes.c_int
        delimiter = Netica.SetMissingDataChar_ns(-1, ctypes.c_void_p(self.cptr))
        checkerr()
        return chr(delimiter)
    
    @case_file_missing_data_char.setter
    def case_file_missing_data_char(self, newchar):
        """The character to use to indicate missing data when creating case files.  
        
        It must be one of asterisk * (42), question mark ? (63), space (32) or 
        absent (0).  
        Like Netica-C SetMissingDataChar_ns.
        """
        if type(newchar) is str:
            newchar = ord(newchar)
        Netica.SetMissingDataChar_ns.restype = ctypes.c_int
        Netica.SetMissingDataChar_ns(ctypes.c_int(newchar), ctypes.c_void_p(self.cptr))
        checkerr()        
   
    def new_file_stream(self, filename, access=None):
        """Create a new Streamer to represent the given file (for reading, writing, etc).  
        
        Like Netica-C NewFileStream_ns.
        """
        if access is not None:
            access = ctypes.c_char_p(access.encode())
        
        Netica.NewFileStream_ns.restype = ctypes.c_void_p
        cptr = Netica.NewFileStream_ns(ctypes.c_char_p(filename.encode()), 
                                                ctypes.c_void_p(self.cptr), access) 
        if dict_initialization:
            if cptr in cptr_dict:
                return cptr_dict[cptr]()
            else:
                return Stream(cptr)
        else:
                return Stream(cptr)
    
    def new_memory_stream(self, name, access=None):
        """Creates a new Streamer to write files in memory.  
        
        Like Netica-C NewMemoryStream_ns.
        """   
        if access is not None:
            access = ctypes.c_char_p(access.encode())         
            
        Netica.NewMemoryStream_ns.restype = ctypes.c_void_p
        cptr = Netica.NewMemoryStream_ns(ctypes.c_char_p(name.encode()), 
                                                  ctypes.c_void_p(self.cptr), access)
        checkerr()
        
        if dict_initialization:
            if cptr in cptr_dict:
                return cptr_dict[cptr]()
            else:
                return Stream(cptr)
        else:
                return Stream(cptr)
            
    def new_net(self, name):
        """Creates a new empty BNet (i.e. Bayesian belief net or decision net).  
        
        Like Netica-C NewNet_bn.
        Create a new instance of the Net class
        """
                     
        Netica.NewNet_bn.restype = ctypes.c_void_p
        cptr = Netica.NewNet_bn(ctypes.c_char_p(name.encode()), 
                                    ctypes.c_void_p(self.cptr))
        checkerr()
        
        return create_net(cptr)
    
    def get_net(self, name):
        """Returns a BNet that Netica currently has in memory.  
        
        Can call funtion with either net name or integer index. If there are
        multiple nets with the same name, return first net of that name. When 
        called with an integer, each value returns a different net, where
        the first is 0. Function returns None if there is no net by the given
        name or there aren't as many nets as the int passed.
        Like Netica-C GetNthNet_bn.
        """ 
        if isinstance(name, int):
            Netica.GetNthNet_bn.restype = ctypes.c_void_p
            cptr = Netica.GetNthNet_bn(ctypes.c_int(name), 
                                        ctypes.c_void_p(self.cptr))
        elif isinstance(name, str):
            Netica.GetNthNet_bn.restype = ctypes.c_void_p
            i = 0
            while True:
                net = Netica.GetNthNet_bn(ctypes.c_int(i), 
                                          ctypes.c_void_p(self.cptr))               
                if net is None:
                    break
                
                # Check the name of the net without creating a net object in python
                Netica.GetNetName_bn.restype = ctypes.c_char_p
                net_name = Netica.GetNetName_bn(ctypes.c_void_p(net)).decode()
                               
                if name == net_name:
                    break
                i = i + 1                           
            cptr = net
        else:
            raise TypeError('An integer or string is required (got type {})'.format(type(name).__name__))
        
        checkerr()
        if cptr:
            return create_net(cptr)

    def new_caseset(self, name):
        """Create an initially empty set of cases.  
        
        'name' argument is not required (i.e. may be empty).  
        Like Netica-C NewCaseset_cs.
        """
        if name is not None:
            name = ctypes.c_char_p(name.encode())
        Netica.NewCaseset_cs.restype = ctypes.c_void_p
        cptr = Netica.NewCaseset_cs(name, ctypes.c_void_p(self.cptr))
        checkerr()
        
        return Caseset(cptr)
    
    def new_learner(self, method):
        """Create a new learner, which will learn using 'method'. 
        
        Method can be one of LearningMethod.COUNTING, LearningMethod.EM, or 
        LearningMethod.GRADIENT_DESCENT.
        Like Netica-C NewLearner_bn.
        """
        Netica.NewLearner_bn.restype = ctypes.c_void_p
        cptr = Netica.NewLearner_bn(ctypes.c_int(method.value), None, 
                                    ctypes.c_void_p(self.cptr))
        checkerr()
        
        return Learner(cptr)
    
    def new_dbmanager(self, connect_str, options):
        """Create a new database manager, and have it connect with database.  
        
        Like Netica-C NewDBManager_cs.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.NewDBManager_cs.restype = ctypes.c_void_p
        cptr = Netica.NewDBManager_cs(ctypes.c_char_p(connect_str.encode()), 
                                      options, ctypes.c_void_p(self.cptr))
        checkerr()
        
        return DatabaseManager(cptr)
        
    def new_random_generator(self, seed, options):
        """Create a new RandomGenerator.
        
        For seed, pass a positive (or zero) integer in the form of a string.
        options should either be None or the string "Nondeterministic".
        Like Netica-C NewRandomGenerator_ns.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.NewRandomGenerator_ns.restype = ctypes.c_void_p
        cptr = Netica.NewRandomGenerator_ns(ctypes.c_char_p(seed.encode()), 
                                                 ctypes.c_void_p(self.cptr), options)
        checkerr()
        
        return RandomGenerator(cptr)
  
    def control_concurrency(self, command, value):
        """Control whether Netica operates single or multi-threaded, and how it 
        does its multi-threading.  
        
        Like Netica-C ControlConcurrency_ns.
        """
        Netica.ControlConcurrency_ns.restype = ctypes.c_char_p
        concurrency = Netica.ControlConcurrency_ns(ctypes.c_char_p(self.cptr), 
                                                   ctypes.c_char_p(command.encode()),
                                                   ctypes.c_char_p(value.encode()))
        checkerr()
        return concurrency.decode()

"""-------------------------------Stream Class------------------------------"""
    
class Stream:     

    def __init__(self, cptr):
               
        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
                                    
    def __del__(self):
        """Remove this Streamer, freeing all resources it consumes, including memory.
            
        Like Netica-C DeleteStream_ns.
        """
        Netica.DeleteStream_ns.restype = None
        Netica.DeleteStream_ns(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None

    '''
    def delete(self):
        """Remove this Streamer, freeing all resources it consumes, including memory.
            
        Like Netica-C DeleteStream_ns.
        """
        Netica.DeleteStream_ns.restype = None
        Netica.DeleteStream_ns(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''    
        
    def set_encryption(self, encryption):
        """Set the encryption that will be used to encrypt/decrypt anything 
        written/read to this stream.  
        
        Pass None to remove the encryption
        Like Netica-C SetStreamEncryption_ns.
        """
        if encryption is not None:
            encryption = ctypes.c_char_p(encryption.encode())
        Netica.SetStreamEncryption_ns.restype = None
        Netica.SetStreamEncryption_ns(ctypes.c_void_p(self.cptr), encryption)
        checkerr()
      
    def set_contents(self, buffer, copy=True):
        """Set the contents of this stream to 'buffer'.
        
        Buffer should be a string or an array of bytes, i.e. unsigned 1 byte 
        integers. 
        Like Netica-C SetStreamContents_ns.
        """
        length = len(buffer)
        if buffer is not None:
            if isinstance(buffer, str):
                buffer = buffer.encode()
            buffer = ctypes.c_char_p(buffer)
        Netica.SetStreamContents_ns.restype = None
        Netica.SetStreamContents_ns(ctypes.c_void_p(self.cptr), buffer,
                                    ctypes.c_longlong(length), ctypes.c_int(copy))
        checkerr()
    
    def get_contents(self, return_as_bytes=False):             
        """Return the contents as set from SetContents, read from file or 
        generated by Netica.  
        
        Like Netica-C GetStreamContents_ns.
        """
        clength = ctypes.c_longlong(0)
        Netica.GetStreamContents_ns.restype = ctypes.POINTER(ctypes.c_ubyte)
        contents_pointer = Netica.GetStreamContents_ns(ctypes.c_void_p(self.cptr), 
                                                       ctypes.byref(clength))
       
        checkerr()
        length = clength.value
        contents = bytearray()
        
        for i in range(length):
            contents.append(contents_pointer[i])
        
        if return_as_bytes is False:
            contents = contents.decode()

        return contents
      
    def read_net(self, options):
        """Read a BNet (Bayes net) from File.  
        
        'options' can be one of \"NoVisualInfo\", \"NoWindow\", or the empty 
        string (means create a regular window). 
        Like Netica-C ReadNet_bn.
        """
        Netica.ReadNet_bn.restype = ctypes.c_void_p
        cptr = Netica.ReadNet_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(options.value))
        checkerr()
        
        return create_net(cptr)


"""--------------------------------Net Class--------------------------------"""

class Net:
    
    def __init__(self, cptr):
          
        self.cptr = cptr
        
        self.python_owner = True
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
        
        elif userdata_initialization:
            cdata = ctypes.cast(ctypes.pointer(ctypes.py_object(self)), ctypes.c_void_p)
            Netica.SetNetUserData_bn.restype = None
            Netica.SetNetUserData_bn(ctypes.c_void_p(self.cptr), 0, cdata)
            checkerr()
        
    def __del__(self):
        """Remove this net, freeing all resources it consumes, including memory.
        
        Like Netica-C DeleteNet_bn.
        """
        if self.python_owner:
            Netica.DeleteNet_bn.restype = None
            Netica.DeleteNet_bn(ctypes.c_void_p(self.cptr))
            checkerr()
        if dict_initialization:
            if self.cptr is not None:
                del cptr_dict[self.cptr]
        self.cptr = None
    
    '''   
    def delete(self):
        """Remove this net, freeing all resources it consumes, including memory.
        
        Like Netica-C DeleteNet_bn.
        """
        Netica.DeleteNet_bn.restype = None
        Netica.DeleteNet_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''

    def copy(self, new_name, new_env, options):
        """Returns a duplicate of this net, but renamed to 'new_name'.  
        
        Options allows you to control what gets copied. It can be any 
        combination of the following strings, separated by commas: "no_nodes", 
        "no_links", "no_tables", and "no_visual".  
        Like Netica-C CopyNet_bn.
        """
        if new_name is not None:
            new_name = ctypes.c_char_p(new_name.encode())
        
        if new_env is not None:
            if not isinstance(new_env, Environ):
                raise TypeError('An Environ is required (got type {})'.format(type(new_env).__name__))
            new_env = ctypes.c_void_p(new_env.cptr)
        
        if options is None:
            options = ''
            
        Netica.CopyNet_bn.restype = ctypes.c_void_p
        cptr = Netica.CopyNet_bn(ctypes.c_void_p(self.cptr), new_name, 
                                 new_env, ctypes.c_char_p(options.encode()))
        checkerr()
        return create_net(cptr)  
      
    def new_node(self, name, states):
        """Add a new node to this net.  
        
        Like Netica-C NewNode_bn.
        """

        if isinstance(states, str):
            num_states = len(states.split(","))
        else:
            num_states = states

        if name is not None:
            name = ctypes.c_char_p(name.encode())

        Netica.NewNode_bn.restype = ctypes.c_void_p
        cptr = Netica.NewNode_bn(name, num_states, ctypes.c_void_p(self.cptr))
        checkerr()

        node = create_node(cptr)

        if isinstance(states, str):
            Netica.SetNodeStateNames_bn.restype = None
            Netica.SetNodeStateNames_bn(ctypes.c_void_p(node.cptr), 
                                                ctypes.c_char_p(states.encode()))
            checkerr()
        
        return node
        
    def copy_nodes(self, nodes, options=None):
        """Duplicate the nodes (and links between them) in the list passed, 
        and return a list of the new nodes.  
        
        The originals may be from a different net.
        options can be any combination of the following strings, 
        separated by commas: "no_links", "no_tables".
        Like Netica-C CopyNodes_bn.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())    
        
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))      
        
        Netica.CopyNodes_bn.restype = ctypes.c_void_p
        cptr = Netica.CopyNodes_bn(ctypes.c_void_p(nodes.cptr), 
                                   ctypes.c_void_p(self.cptr), options)
        checkerr()
        
        return create_nodelist(cptr)

    @property
    def file_name(self):
        """Return the file name, including directory, that this net was last 
        read from or written to.  
        
        Like Netica-C GetNetFileName_bn.
        """
        Netica.GetNetFileName_bn.restype = ctypes.c_char_p
        file_name = Netica.GetNetFileName_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return file_name.decode()     

    @property
    def name(self):
        """Name of this net (restricted to 30 character alphanumeric).  
        
        See also 'Title' property.  
        Like Netica-C G/SetNetName_bn.
        """
        Netica.GetNetName_bn.restype = ctypes.c_char_p
        name = Netica.GetNetName_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return name.decode()

    @name.setter
    def name(self, name):
        """Name of this net (restricted to 30 character alphanumeric).  
        
        See also 'Title' property.
        Like Netica-C G/SetNetName_bn.
        """
        Netica.SetNetName_bn.restype = None
        Netica.SetNetName_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(name.encode()))
        checkerr()
    
    @property
    def title(self):
        """Unrestricted label for this net.  
        
        See also 'Name' property.  
        Like Netica-C G/SetNetTitle_bn.
        """
        Netica.GetNetTitle_bn.restype = ctypes.c_char_p
        title = Netica.GetNetTitle_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return title.decode()

    @title.setter
    def title(self, title):
        """Unrestricted label for this net.  
        
        See also 'Name' property.  
        Like Netica-C G/SetNetTitle_bn.
        """
        Netica.SetNetTitle_bn.restype = None
        Netica.SetNetTitle_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(title.encode()))
        checkerr()    
    
    @property
    def comment(self):
        """Documentation or description of this net.
        
        Like Netica-C G/SetNetComment_bn.
        """
        Netica.GetNetComment_bn.restype = ctypes.c_char_p
        comment = Netica.GetNetComment_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return comment.decode()

    @comment.setter
    def comment(self, comment):
        """Documentation or description of this net.
        
        Like Netica-C G/SetNetComment_bn.
        """
        Netica.SetNetComment_bn.restype = None
        Netica.SetNetComment_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(comment.encode()))
        checkerr()
  
    @property
    def elimination_order(self):
        """Ordered list of all the nodes in this net (except constant and utility 
        nodes), used to guide compiling to find an efficient junction tree.
        
        Getter creates a constant NodeList, i.e. one that cannot be 
        modified or deleted.  To make modifications, duplicate this Nodelist, 
        using NodeList.copy.  Funtion returns None if there is no order 
        currently associated with the net
        Like Netica-C G/SetNetElimOrder_bn.
        """
        Netica.GetNetElimOrder_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNetElimOrder_bn(ctypes.c_void_p(self.cptr)) 
        if cptr:
            return create_nodelist(cptr, is_const=True)
        else:
            return None

    @elimination_order.setter
    def elimination_order(self, elim_order):
        """Ordered list of all the nodes in this net (except constant and utility 
        nodes), used to guide compiling to find an efficient junction tree.
        
        Getter creates a constant NodeList, i.e. one that cannot be 
        modified or deleted.  To make modifications, duplicate this Nodelist, 
        using NodeList.copy.  Funtion returns None if there is no order 
        currently associated with the net
        Like Netica-C G/SetNetElimOrder_bn.
        """
        Netica.SetNetElimOrder_bn.restype = ctypes.c_int
        Netica.SetNetElimOrder_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(elim_order.cptr))
        checkerr()
         
    @property
    def autoupdate(self):
        """Turn auto-update on (1) or off (0).
        
        May be computationally heavy if an auto update is needed.
        If =1 then beliefs are automatically recalculated whenever a finding 
        is entered into this net, if =0 they aren't.  
        Like Netica-C G/SetNetAutoUpdate_bn. 
        """
        Netica.GetNetAutoUpdate_bn.restype = ctypes.c_int
        autoupdate_state = Netica.GetNetAutoUpdate_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return autoupdate_state  
    
    @autoupdate.setter
    def autoupdate(self, auto_update):           
        """Turn auto-update on (1) or off (0).
        
        May be computationally heavy if an auto update is needed.
        If =1 then beliefs are automatically recalculated whenever a finding 
        is entered into this net, if =0 they aren't.  
        Like Netica-C G/SetNetAutoUpdate_bn.
        """
        Netica.SetNetAutoUpdate_bn.restype = ctypes.c_int
        Netica.SetNetAutoUpdate_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(auto_update))
        checkerr()
        
    def set_user_field(self, name, data, kind=0):
        """Attach user defined data to this net under category 'field_name'.  
        
        Like Netica-C SetNetUserField_bn.
        """
        if isinstance(data, str):
            data = data.encode()
        if isinstance(data, int):
            data = str(data).encode()
        length = len(data)
        Netica.SetNetUserField_bn.restype = None
        Netica.SetNetUserField_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(name.encode()),
                                  ctypes.c_char_p(data), ctypes.c_int(length), ctypes.c_int(kind))
        checkerr()      

    def get_user_field(self, name, return_type=None, kind=0):
        """Return user defined data to this net under category 'field_name'. 
        
        return_type can be set to 'str' or 'int' to convert the user field from
        a bytearray to a string or integer.
        Like Netica-C GetNetUserField_bn.
        """
        clength = ctypes.c_int(0)
        Netica.GetNetUserField_bn.restype = ctypes.POINTER(ctypes.c_ubyte)
        data_pointer = Netica.GetNetUserField_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(name.encode()),
                                                 ctypes.byref(clength), ctypes.c_int(kind))
        checkerr()
        length = clength.value
        data = bytearray()
        
        for i in range(length):
            data.append(data_pointer[i])
        
        if return_type is 'str':
            data = data.decode()
        if return_type is 'int':
            data = int(data)
        
        return data

    def get_nth_user_field(self, index, kind=0):
        """Return the Nth element of user defined data from this net, and its 
        category 'field_name'.  
        
        Like Netica-C GetNetNthUserField_bn.
        """
        clength = ctypes.c_int(0)
        cname = ctypes.c_char_p(b'')
        cvalue = ctypes.pointer(ctypes.c_ubyte(0))
        Netica.GetNetNthUserField_bn.restype = None
        Netica.GetNetNthUserField_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(index),
                                     ctypes.byref(cname), ctypes.byref(cvalue),
                                     ctypes.byref(clength), ctypes.c_int(kind))
        checkerr()
        length = clength.value
        name = cname.value
        data = bytearray()
        
        for i in range(length):
            data.append(cvalue[i])
        
        return name, data

    @property
    def nodes(self):
        """Return a list consisting of all the nodes in this net.  
        
        Creates a constant NodeList, i.e. one that cannot be 
        modified or deleted.  To make modifications, duplicate this Nodelist, 
        using NodeList.copy.
        Like Netica-C GetNetNodes2_bn.
        """
        Netica.GetNetNodes2_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNetNodes2_bn(ctypes.c_void_p(self.cptr), None)
        checkerr()
  
        return create_nodelist(cptr, is_const=True)
    
    def get_nodes(self, options):
        """Return a list consisting of all the nodes in this net.  
        
        Creates a constant NodeList, i.e. one that cannot be 
        modified or deleted.  To make modifications, duplicate this Nodelist, 
        using NodeList.copy.
        Like Netica-C GetNetNodes2_bn.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.GetNetNodes2_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNetNodes2_bn(ctypes.c_void_p(self.cptr), options)
        checkerr()
        
        return create_nodelist(cptr, is_const=True)

    def get_related_nodes(self, related_nodes, relation, nodes):
        """Put in 'related_nodes' those nodes that bear 'relation' to any of 
        the nodes in 'nodes'.  
        
        See also Node.get_related_nodes.  
        Like Netica-C GetRelatedNodesMult_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        if not isinstance(related_nodes, NodeList):
            raise TypeError('A Nodelist is required (got type {})'.format(type(related_nodes).__name__))
        
        Netica.GetRelatedNodesMult_bn.restype = None
        Netica.GetRelatedNodesMult_bn(ctypes.c_void_p(related_nodes.cptr), 
                                      ctypes.c_char_p(relation.encode()), 
                                      ctypes.c_void_p(nodes.cptr))
        checkerr()
   
    def write(self, file):
        """"Write this net to the indicated file (which may include directory path).  
        
        Like Netica-C WriteNet_bn.
        """
        if not isinstance(file, Stream):
            raise TypeError('A Stream is required (got type {})'.format(type(file).__name__))
            
        Netica.WriteNet_bn.restype = None
        Netica.WriteNet_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(file.cptr))
        checkerr()    
        
    def write_findings(self, nodes, file, ID_num, freq):
        """Write a case from the node list to the given stream.
        
        Also writes the case's ID number and frequency, if those parameters are 
        passed.  
        Like Netica-C WriteNetFindings_bn.
        """
        if nodes is not None:
            if not isinstance(nodes, NodeList):
                raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
            nodes = ctypes.c_void_p(nodes.cptr)
            
        if not isinstance(file, Stream):
            raise TypeError('A Stream is required (got type {})'.format(type(file).__name__))
            
        Netica.WriteNetFindings_bn.restype = ctypes.c_longlong
        Netica.WriteNetFindings_bn(nodes, ctypes.c_void_p(file.cptr), 
                                   ctypes.c_longlong(ID_num), ctypes.c_double(freq))
        checkerr()
    
    def read_findings(self, case_posn, stream, add, nodes, ID_num, freq):
        """Read a case from the given stream into the node list.
        
        Also reads and returns the case's ID number and frequency, if that is 
        present.  
        Like Netica-C ReadNetFindings2_bn.
        """
        if case_posn is not None:
            c_case_posn = ctypes.c_longlong(case_posn.value)
            case_posn_ref = ctypes.pointer(c_case_posn)
        else:
            case_posn_ref = case_posn
        
        if ID_num is not None:
            c_ID_num = ctypes.c_longlong(ID_num)
            ID_num_ref = ctypes.pointer(c_ID_num)
        else:
            ID_num_ref = ID_num
        
        if freq is not None:
            c_freq = ctypes.c_double(freq)
            freq_ref = ctypes.pointer(c_freq)
        else:
            freq_ref = freq
        
        if not isinstance(stream, Stream):
            raise TypeError('A Stream is required (got type {})'.format(type(stream).__name__))
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        
        Netica.ReadNetFindings2_bn.restype = None
        Netica.ReadNetFindings2_bn(case_posn_ref, ctypes.c_void_p(stream.cptr), 
                                   ctypes.c_ubyte(add), ctypes.c_void_p(nodes.cptr),
                                   ID_num_ref, freq_ref)
        checkerr()
        
        if case_posn is not None:
            case_posn = c_case_posn.value
        if ID_num is not None:
            ID_num = c_ID_num.value
        if freq is not None:
            freq = c_freq.value
        
        return case_posn, ID_num, freq
    
    def new_nodelist(self, length):
        """Create a new empty NodeList, i.e. list of Nodes.  
        
        It will be initialized to 'nodes', which should be an array of BNodes, 
        or empty.  
        Like Netica-C NewNodeList2_bn
        """
        Netica.NewNodeList2_bn.restype = ctypes.c_void_p
        cptr = Netica.NewNodeList2_bn(ctypes.c_int(length), 
                                      ctypes.c_void_p(self.cptr))
        checkerr()
       
        return create_nodelist(cptr)
    
    def map_state_list(self, src_states, src_nodes, dest_states, dest_nodes):
        """***nodocs
        """
        if not isinstance(src_nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(src_nodes).__name__))
        if not isinstance(dest_nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(dest_nodes).__name__))
        src_length = len(src_states)
        csrc_states = (ctypes.c_int*src_length)(*src_states)
        dest_length = len(dest_states)
        cdest_states = (ctypes.c_int*dest_length)(*dest_states)
        
        Netica.MapStateList2_bn.restype = None
        Netica.MapStateList2_bn(csrc_states, ctypes.c_int(src_length), 
                                ctypes.c_void_p(src_nodes.cptr), cdest_states,
                                ctypes.c_int(dest_length), ctypes.c_void_p(dest_nodes))
        checkerr()

    
    def revise_CPTs_by_finding(self, nodes, degree, updating=0):
        """Revise the CPTs of these nodes to account for the currently entered 
        case, i.e. findings. 
        
        Like Netica-C ReviseCPTsByFindings_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        Netica.ReviseCPTsByFindings_bn.restype = None
        Netica.ReviseCPTsByFindings_bn(ctypes.c_void_p(nodes.cptr), ctypes.c_int(updating),
                                       ctypes.c_double(degree))
        checkerr()

    def revise_CPTs_by_case_file(self, file, nodes, degree, updating=0):
        """Revise the CPTs of these nodes, to account for the cases in the given file.
        
        Like Netica-C ReviseCPTsByCaseFile_bn.
        """
        if not isinstance(file, Stream):
            raise TypeError('A Stream is required (got type {})'.format(type(file).__name__))
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        
        Netica.ReviseCPTsByCaseFile_bn.restype = None
        Netica.ReviseCPTsByCaseFile_bn(ctypes.c_void_p(file.cptr), ctypes.c_void_p(nodes.cptr), 
                                       ctypes.c_int(updating), ctypes.c_double(degree))
        checkerr()

    def absorb_nodes(self, nodes):
        """Absorbs node, so that they are removed from this net, but the net's 
        overall joint probabilities for the remaining nodes is unchanged.  
        
        nodes should be a modifyable NodeList.
        Like Netica-C AbsorbNodes_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
            
        Netica.AbsorbNodes_bn.restype = None
        Netica.AbsorbNodes_bn(ctypes.c_void_p(nodes.cptr))
        checkerr()   
        
        # Create a new empty nodelist to replace the damaged one
        Netica.NewNodeList2_bn.restype = ctypes.c_void_p
        new_cptr = Netica.NewNodeList2_bn(ctypes.c_int(0), 
                                          ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            weakref = cptr_dict[nodes.cptr]
            del cptr_dict[nodes.cptr]
            cptr_dict[new_cptr] = weakref
        nodes.cptr = new_cptr

    def retract_findings(self):
        """Retract all findings entered in this net (from all nodes except 'constant' nodes).
        
        Like Netica-C RetractNetFindings_bn.
        """
        Netica.RetractNetFindings_bn.restype = None
        Netica.RetractNetFindings_bn(ctypes.c_void_p(self.cptr))
        checkerr()

    def compile_net(self):                      
        """Compile this net for fast inference.
        
        Like Netica-C CompileNet_bn.
        """
        Netica.CompileNet_bn.restype = None
        Netica.CompileNet_bn(ctypes.c_void_p(self.cptr))
        checkerr()

    def uncompile(self):
        """Reverse the effect of the Compile function.  
        
        Used just to reduce memory consumption of a net, by releasing the 
        junction tree, etc.  
        Like Netica-C UncompileNet_bn.
        """
        Netica.UncompileNet_bn.restype = None
        Netica.UncompileNet_bn(ctypes.c_void_p(self.cptr))
        checkerr()

    def get_size_compiled(self, method=0):
        """Return the number of bytes required for the internal structures, 
        including junction tree, of a compiled net.  
        
        Maximum inference time is linearly related to this quantity.  
        Like Netica-C SizeCompiledNet_bn.
        """
        Netica.SizeCompiledNet_bn.restype = ctypes.c_double
        size = Netica.SizeCompiledNet_bn(ctypes.c_void_p(self.cptr), 
                                         ctypes.c_int(method))
        checkerr()
        return size

    def findings_probability(self):                     # May want to make this a property
        """Return the joint probability of all the findings entered so far. 
        
        See also get_joint_probability.
        Like Netica-C FindingsProbability_bn.
        """
        Netica.FindingsProbability_bn.restype = ctypes.c_double
        prob = Netica.FindingsProbability_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return prob    

    def get_expected_utility(self):
        """Return the overall expected utility for all the utility nodes.
        
        Assumes an optimal strategy is followed, given any findings entered.  
        Like Netica-C GetNetExpectedUtility_bn.
        """
        Netica.GetNetExpectedUtility_bn.retype = ctypes.c_float
        utility = Netica.GetNetExpectedUtility_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return utility

    def get_joint_probability(self, nodes, states):
        """Return the joint probability that these nodes are in the passed 
        node states, given the findings currently entered in the net.  
        
        states must have one entry for each node in the BNodes, and in the same 
        order.  
        Like Netica-C GetJointProbability_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        length = len(states)
        cstates = (ctypes.c_int*length)(*states)
        Netica.GetJointProbability_bn.restype = ctypes.c_double
        joint_prob = Netica.GetJointProbability_bn(ctypes.c_void_p(nodes.cptr), 
                                                   cstates, ctypes.c_int(length)) 
        checkerr()
        return joint_prob

    def get_most_probable_config(self, nodes, nth=0):
        """Return an array of states, one for each node, that indicates the most 
        probable configuration.  
        
        Currently only works if 'nodes' was created with the NodeList returned
        from Net.nodes. 
        Like Netica-C MostProbableConfig2_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        length = nodes.length
        config = [0] * length
        cconfig = (ctypes.c_int*length)(*config) 
        clength = ctypes.c_int(length)
        Netica.MostProbableConfig2_bn.restype = None
        Netica.MostProbableConfig2_bn(ctypes.c_void_p(nodes.cptr), ctypes.byref(cconfig), 
                                      ctypes.byref(clength), nth)
        checkerr()
        
        ret_length = clength.value
        for i in range(ret_length):
            config[i] = cconfig[i]
        
        return config

    def generate_random_case(self, nodes, method, num, rand):    
        """Generate a random case by simulation.
        
        Enters findings into the passed node list.  
        Like Netica-C GenerateRandomCase_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        if num is None:
            num = 0
        if rand is not None:
            if not isinstance(rand, RandomGenerator):
                raise TypeError('A RandomGenerator is required (got type {})'.format(type(rand).__name__))
            rand = ctypes.c_void_p(rand.cptr)
        Netica.GenerateRandomCase_bn.restype = ctypes.c_int
        res = Netica.GenerateRandomCase_bn(ctypes.c_void_p(nodes.cptr), 
                                           ctypes.c_int(method.value), 
                                           ctypes.c_double(num), rand)
        checkerr()
        return res

    def get_all_nodesets(self, include_system, vis=None):
        """Return a comma-separated list of all node-set names.  
        
        Like Netica-C GetAllNodesets_bn.
        """
        if vis is not None:
            vis = ctypes.c_void_p(vis)
        Netica.GetAllNodesets_bn.restype = ctypes.c_char_p
        nodesets = Netica.GetAllNodesets_bn(ctypes.c_void_p(self.cptr), 
                                            ctypes.c_bool(include_system), vis)
        checkerr()
        return nodesets.decode()

    def get_nodeset_color(self, nodeset, vis=None):
        """Get the color of the given node-set as an integer.
        
        Represented as 8 bits red, 8 bits green, 8 bits blue.  To convert the 
        returned integer to RGB values, one can use:
            Blue =  RGBint & 255
            Green = (RGBint >> 8) & 255
            Red =   (RGBint >> 16) & 255
        Like Netica-C SetNodesetColor_bn.
        """
        if vis is not None:
            vis = ctypes.c_void_p(vis)
        Netica.SetNodesetColor_bn.restype = ctypes.c_int
        color = Netica.SetNodesetColor_bn(ctypes.c_char_p(nodeset.encode()), 
                                          ctypes.c_int(QUERY_ns),
                                          ctypes.c_void_p(self.cptr), vis)
        checkerr()
        return color
    
    def set_nodeset_color(self, nodeset, color, vis=None):
        """Set the color of the given node-set as an integer.
        
        Represented as 8 bits red, 8 bits green, 8 bits blue.  To convert to an 
        integer, use the formula n = r256^2 + g256 + b.
        Like Netica-C SetNodesetColor_bn.
        """
        if vis is not None:
            vis = ctypes.c_void_p(vis)
        Netica.SetNodesetColor_bn.restype = ctypes.c_int
        Netica.SetNodesetColor_bn(ctypes.c_char_p(nodeset.encode()), ctypes.c_int(color),
                                  ctypes.c_void_p(self.cptr), vis)
        checkerr()

    def reorder_nodesets(self, nodeset_order, vis=None):
        """Make the node-sets in the comma-delimited list passed the highest priority ones.
        
        The first in the list is the very highest priority.  To restore the old 
        setting, pass it the list returned by GetAllNodesets.  
        Like Netica-C ReorderNodesets_bn.
        """
        if vis is not None:
            vis = ctypes.c_void_p(vis)
        Netica.ReorderNodesets_bn.restype = None
        Netica.ReorderNodesets_bn(ctypes.c_void_p(self.cptr), 
                                  ctypes.c_char_p(nodeset_order.encode()), vis)
        checkerr()
    
    def new_tester(self, test_nodes, unobsv_nodes, tests):
        """Create a Tester to performance test this net using a set of cases.  
        
        Like Netica-C NewNetTester_bn.
        """
        if not isinstance(test_nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(test_nodes).__name__))
        if not isinstance(unobsv_nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(unobsv_nodes).__name__))
        Netica.NewNetTester_bn.restype = ctypes.c_void_p
        cptr = Netica.NewNetTester_bn(ctypes.c_void_p(test_nodes.cptr),
                                      ctypes.c_void_p(unobsv_nodes.cptr), tests)
        checkerr()
        
        return Tester(cptr)

    def set_num_undos_kept(self, count_limit, memory_limit, options=None):
        """Turn on the ability to undo operations, and set how many to save in memory.  
        
        Like Netica-C SetNetNumUndos_bn.
        """
        if count_limit is None:
            count_limit = -1
        if memory_limit is None:
            memory_limit = -1
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.SetNetNumUndos_bn.restype = None
        Netica.SetNetNumUndos_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(count_limit),
                                 ctypes.c_double(memory_limit), options)

    def undo_last_operation(self, to_when=-1):
        """Undoes the last operation done on this net, whether it was done by program or GUI user.  
        
        Call this repeatedly to undo earlier operations.  Pass -1 for to_when.  
        See also RedoOperation().  
        Like Netica-C UndoNetLastOper_bn.
        """
        Netica.UndoNetLastOper_bn.restype = ctypes.c_int
        succeeded = Netica.UndoNetLastOper_bn(ctypes.c_void_p(self.cptr),
                                              ctypes.c_longlong(to_when))
        checkerr()
        return succeeded

    def redo_operation(self, to_when=-1):
        """Re-does an operation that was undone with UndoLastOperation(), if there is one.  
        
        May be called repeatedly to redo several operations.  Pass -1 for 
        to_when.  
        Like Netica-C RedoNetOper_bn.
        """
        Netica.RedoNetOper_bn.restype = ctypes.c_int
        succeeded = Netica.RedoNetOper_bn(ctypes.c_void_p(self.cptr),
                                          ctypes.c_longlong(to_when))
        checkerr()
        return succeeded

    def create_custom_report(self, sel_nodes, templat, options=None):
        """Create a text or HTML report on nodes or the overall net, based on 
        the template passed in.  
        
        Like Netica-C CreateCustomReport_bn.
        """
        if sel_nodes is not None:
            if not isinstance(sel_nodes, NodeList):
                raise TypeError('A NodeList is required (got type {})'.format(type(sel_nodes).__name__))
            sel_nodes = ctypes.c_void_p(sel_nodes.cptr)
        if options is not None:
            options = ctypes.c_char_p(options.encode())
            
        Netica.CreateCustomReport_bn.restype = ctypes.c_char_p
        report = Netica.CreateCustomReport_bn(ctypes.c_void_p(self.cptr), sel_nodes, 
                                              ctypes.c_char_p(templat.encode()), options)
        checkerr()
        return report.decode()
            
    def expand_time_series(self, result_time, burn_time, dimn=0, options=None):
        """Create a time-expanded version of this dynamic Bayes net, showing 
        time slices for the given time range.  
        
        Like Netica-C ExpandNet_bn.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.ExpandNet_bn.restype = ctypes.c_void_p
        cptr = Netica.ExpandNet_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(dimn),
                            ctypes.c_double(result_time), ctypes.c_double(burn_time),
                            options)
        checkerr()
        
        return create_net(cptr)

    def get_node_at_time(self, name, time):
        """Returns the node of this expanded dynamic Bayes net (DBN) whose name 
        starts with 'name' and whose value is valid at the point in time 'time' 
        (which is an array of length one consisting of a single number, the time).  
        
        Like Netica-C GetNodeAtTime_bn.
        """
        # Currently untested - not in wide use
        Netica.GetNodeAtTime_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNodeAtTime_bn(ctypes.c_void_p(self.cptr), 
                                       ctypes.c_char_p(name.encode()), 
                                       ctypes.c_double(time))
        checkerr()
        
        return create_node(cptr)  
    
    def set_random_gen(self, rand, is_private):
        """***nodocs
        """
        if not isinstance(rand, RandomGenerator):
            raise TypeError('A RandomGenerator is required (got type {})'.format(type(rand).__name__))
        Netica.SetNetRandomGen_bn.restype = None
        Netica.SetNetRandomGen_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(rand.cptr),
                                  ctypes.c_bool(is_private))
        checkerr()

    def get_node_named(self, name):               # May want to rename to get_node 
        """Return a node from this net by name.  
        
        Like Netica-C GetNodeNamed_bn. 
        """
        Netica.GetNodeNamed_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNodeNamed_bn(ctypes.c_char_p(name.encode()), 
                                      ctypes.c_void_p(self.cptr))
        checkerr()
        
        if cptr:
            return create_node(cptr)            
        else:
            return None

    def list_to_nodelist(self, list_of_nodes):
        """Convert a Python List of nodes into a NodeList with the same ordering.
        
        Can pass in a list of nodes, node names, or some combination of the two.
        """
        #PythonOnly
        nodelist = self.new_nodelist(len(list_of_nodes))
        for idx, val in enumerate(list_of_nodes):           
            if isinstance(val, str):
                val = self.get_node_named(val)            
            nodelist.set_nth_node(val, idx)        
        return nodelist


"""--------------------------------Node Class-------------------------------"""    
     
class Node:
    
    def __init__(self, cptr):
               
        self.cptr = cptr
        
        self.deleted_by_netica = False
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
        
        elif userdata_initialization:
            cdata = ctypes.cast(ctypes.pointer(ctypes.py_object(self)), ctypes.c_void_p)
            Netica.SetNodeUserData_bn.restype = None
            Netica.SetNodeUserData_bn(ctypes.c_void_p(self.cptr), 0, cdata)
            checkerr()
    
    def __del__(self):
        """Remove this node, freeing all resources it consumes, including memory. 
        
        Like Netica-C DeleteNode_bn.
        """
        #PythonOnly
        if dict_initialization:
            if self.cptr is not None:
                if self.cptr in cptr_dict:
                    del cptr_dict[self.cptr]
        self.cptr = None
    
    
    def delete(self):
        """Remove this node, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteNode_bn.
        """
        Netica.DeleteNode_bn.restype = None
        Netica.DeleteNode_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            if self.cptr is not None:
                del cptr_dict[self.cptr]
        self.cptr = None
        
    @property
    def net(self):
        """Return the BNet that this node is part of.  
        
        Like Netica-C GetNodeNet_bn.
        """        
        Netica.GetNodeNet_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNodeNet_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        
        return create_net(cptr)
    
    @property
    def name(self):
        """Name of this node.
        
        Restricted to 30 character alphanumeric.      
        See also 'Title' property.  
        Like Netica-C G/SetNodeName_bn
        """
        Netica.GetNodeName_bn.restype = ctypes.c_char_p
        name = Netica.GetNodeName_bn(ctypes.c_void_p(self.cptr)).decode()
        checkerr()
        return name
    
    @name.setter
    def name(self, name):
        """Name of this node.
        
        Restricted to 30 character alphanumeric.      
        See also 'Title' property.  
        Like Netica-C G/SetNodeName_bn
        """
        Netica.SetNodeName_bn.restype = None
        Netica.SetNodeName_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(name.encode()))
        checkerr()
    
    @property
    def title(self):
        """Unrestricted label for this node.  

        Like Netica-C G/SetNodeTitle_bn.
        """
        Netica.GetNodeTitle_bn.restype = ctypes.c_char_p
        title = Netica.GetNodeTitle_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return title.decode()
    
    @title.setter
    def title(self, title):
        """Unrestricted label for this node.  

        Like Netica-C G/SetNodeTitle_bn.
        """
        Netica.SetNodeTitle_bn.restype = None
        Netica.SetNodeTitle_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(title.encode()))
        checkerr()
        
    @property
    def comment(self):
        """Documentation or description of this node.  
        
        Like Netica-C G/SetNodeComment_bn.
        """
        Netica.GetNodeComment_bn.restype = ctypes.c_char_p
        comment = Netica.GetNodeComment_bn(ctypes.c_void_p(self.cptr)).decode()
        checkerr()
        return comment
    
    @comment.setter
    def comment(self, comment):
        """Documentation or description of this node.  
        
        Like Netica-C G/SetNodeComment_bn.
        """
        Netica.SetNodeComment_bn.restype = None
        Netica.SetNodeComment_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(comment.encode()))
        checkerr()

    @property
    def state_numbers(self):
        """***nodocs
        """
        clength = ctypes.c_int(0)
        Netica.GetNodeStateNumbers_bn.restype = ctypes.POINTER(ctypes.c_double)
        state_nums_pointer = Netica.GetNodeStateNumbers_bn(ctypes.c_void_p(self.cptr),
                                                           ctypes.byref(clength))
        checkerr()
        length = clength.value
        if state_nums_pointer:
            state_numbers = []
            for i in range(length):
                state_numbers.extend([state_nums_pointer[i]])
        else:
            state_numbers = None
        return state_numbers

    @state_numbers.setter
    def state_numbers(self, levels):
        """***nodocs
        """
        numstates = len(levels)
        Netica.SetNodeStateNumbers_bn.restype = None
        Netica.SetNodeStateNumbers_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(numstates),
                                      (ctypes.c_double*len(levels))(*levels))
        checkerr()

    @property
    def intervals(self):
        """***nodocs
        """
        clength = ctypes.c_int(0)
        Netica.GetNodeIntervals_bn.restype = ctypes.POINTER(ctypes.c_double)
        intervals_pointer = Netica.GetNodeIntervals_bn(ctypes.c_void_p(self.cptr),
                                                       ctypes.byref(clength))
        checkerr()
        length = clength.value
        if intervals_pointer:
            intervals = []
            for i in range(length):
                intervals.extend([intervals_pointer[i]])
        else:
            intervals = None
        return intervals

    @intervals.setter
    def intervals(self, levels):
        """***nodocs
        """
        numstates = len(levels) - 1
        Netica.SetNodeIntervals_bn.restype = None
        Netica.SetNodeIntervals_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(numstates),
                                   (ctypes.c_double*len(levels))(*levels))
        checkerr()
 
    '''
    def set_state_levels(self, levels):
        """Set state level.
        
        If this node is for a continuous variable, then this is a discretization 
        threshold, otherwise it is a state-number.  
        Like Netica-C SetNodeLevels_bn.
        """
        if self.node_type is 'DISCRETE':
            numstates = len(levels)
        elif self.node_type is 'CONTINUOUS':
            numstates = len(levels) - 1
        else:
            raise NeticaPyError("Netica Error 6500: Node must be discrete or continuos")
        Netica.SetNodeLevels_bn.restype = None
        Netica.SetNodeLevels_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(numstates), 
                                (ctypes.c_double*len(levels))(*levels))
        checkerr()  
       '''     
   
    @property
    def node_type(self):
        """Return whether this node is for a discrete or continuous variable.  
        
        If you want to detect discrete variables and continuous variables that 
        have been discretized, instead use num_states property not-equal zero.  
        Like Netica-C GetNodeType_bn.
        """
        Netica.GetNodeType_bn.restype = ctypes.c_int
        node_type = Netica.GetNodeType_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return NodeType(node_type).name
 
    @property
    def kind(self):
        """How this node is being used (nature, decision, etc).  
        
        Like Netica-C G/SetNodeKind_bn.
        """
        Netica.GetNodeKind_bn.restype = ctypes.c_int
        node_kind = NodeKind(Netica.GetNodeKind_bn(ctypes.c_void_p(self.cptr))).name
        checkerr()
        return node_kind
    
    @kind.setter
    def kind(self, kind):
        """How this node is being used (nature, decision, etc).  
        
        Like Netica-C G/SetNodeKind_bn.
        """
        Netica.SetNodeKind_bn.restype = None
        Netica.SetNodeKind_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(kind.value))
        checkerr()
       
    @property
    def num_states(self):
        """Return the number of states this node has. 
        
        0 for undiscretized continuous nodes.  
        Like Netica-C GetNodeNumberStates_bn.
        """
        Netica.GetNodeNumberStates_bn.restype = ctypes.c_int
        node_num_states = Netica.GetNodeNumberStates_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return node_num_states 

    def set_state_name(self, state, statename):
        """Set the name of this node's state identified by the given index.
        
        Restricted to 30 character alphanumeric.  
        See also StateTitle.  
        Like Netica-C SetNodeStateName_bn.
        """
        Netica.SetNodeStateName_bn.restype = None
        Netica.SetNodeStateName_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state), 
                                   ctypes.c_char_p(statename.encode()))
        checkerr()
        
    def get_state_name(self, state):
        """Return the name of this node's state identified by the given index.
        
        Restricted to 30 character alphanumeric.  
        See also StateTitle.  
        Like Netica-C GetNodeStateName_bn.
        """
        Netica.GetNodeStateName_bn.restype = ctypes.c_char_p
        statename = Netica.GetNodeStateName_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state)).decode()
        checkerr()
        return statename
     
    @property
    def state_names(self):
        """The names of this node's states separated by commas.
        
        See also get_state_name/set_state_name (just one state).  
        Like Netica-C SetNodeStateNames_bn.
        """
        statenames = []
        Netica.GetNodeStateName_bn.restype = ctypes.c_char_p
        for i in range(self.num_states):
            statenames.append(Netica.GetNodeStateName_bn(ctypes.c_void_p(self.cptr), i).decode())
        checkerr()
        statenames = ", ".join(statenames)
        return statenames
    
    @state_names.setter
    def state_names(self, statenames):
        """The names of this node's states separated by commas.
        
        See also get_state_name/set_state_name (just one state).  
        Like Netica-C SetNodeStateNames_bn.
        """
        if statenames is not None:
            statenames = ctypes.c_char_p(statenames.encode())
        Netica.SetNodeStateNames_bn.restype = None
        Netica.SetNodeStateNames_bn(ctypes.c_void_p(self.cptr), statenames)
        checkerr()
    
    def set_state_title(self, state, title):
        """Set the unrestricted title of this node's state identified by the given index.  
        
        See also StateName.  
        Like Netica-C SetNodeStateTitle_bn.
        """
        Netica.SetNodeStateTitle_bn.restype = None
        Netica.SetNodeStateTitle_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state), 
                                   ctypes.c_char_p(title.encode()))
        checkerr()      
    
    def get_state_title(self, state):
        """Get the unrestricted title of this node's state identified by the given index.  
        
        See also StateName.  
        Like Netica-C GetNodeStateTitle_bn.
        """
        Netica.GetNodeStateTitle_bn.restype = ctypes.c_char_p
        statetitle = Netica.GetNodeStateTitle_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state)).decode()
        checkerr()
        return statetitle
    
    def set_state_comment(self, state, comment):
        """Set the comment of this node's state identified by the given index.  
        
        Like Netica-C SetNodeStateComment_bn.
        """
        Netica.SetNodeStateComment_bn.restype = None
        Netica.SetNodeStateComment_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state), 
                                   ctypes.c_char_p(comment.encode()))
        checkerr()      
    
    def get_state_comment(self, state):
        """Return the comment of this node's state identified by the given index.  
        
        Like Netica-C GetNodeStateComment_bn.
        """
        Netica.GetNodeStateComment_bn.restype = ctypes.c_char_p
        statecomment = Netica.GetNodeStateComment_bn(ctypes.c_void_p(self.cptr), 
                                                     ctypes.c_int(state)).decode()
        checkerr()
        return statecomment
    
    def get_state_named(self, state):
        """Get the index of the state with the given name. 
        
        Warning: other nodes with this state name may use a different index.  
        Like Netica-C GetStateNamed_bn.
        """
        Netica.GetStateNamed_bn.restype = ctypes.c_int
        statenum = Netica.GetStateNamed_bn(ctypes.c_char_p(state.encode()), 
                                           ctypes.c_void_p(self.cptr))
        checkerr()
        if statenum is -3:
            return State(statenum).name
        else:
            return statenum
    
    @property
    def parents(self):
        """Return list of the parent nodes of this node.  
        
        That is, those nodes which are at the beginning of a link that enters 
        this node.  
        Creates a constant NodeList, i.e. one that cannot be 
        modified or deleted.  To make modifications, duplicate this Nodelist, 
        using NodeList.copy.
        Like Netica-C GetNodeParents_bn.
        """
        Netica.GetNodeParents_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNodeParents_bn(ctypes.c_void_p(self.cptr)) 
        checkerr()
        
        return create_nodelist(cptr, is_const=True)
    
    @property
    def children(self):
        """Return list of the child nodes of this node.  
        
        That is, those nodes which are at the end of a link that exits this node.  
        Creates a constant NodeList, i.e. one that cannot be 
        modified or deleted.  To make modifications, duplicate this Nodelist, 
        using NodeList.copy.
        Like Netica-C GetNodeChildren_bn.
        """
        Netica.GetNodeChildren_bn.restype = ctypes.c_void_p
        cptr = Netica.GetNodeChildren_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        
        return create_nodelist(cptr, is_const=True)  
    
    def set_input_name(self, link_index, name):
        """Set the name of this node's input (i.e. link) identified by the given index.  
        
        Like Netica-C SetNodeInputName_bn.
        """
        Netica.SetNodeInputName_bn.restype = None
        Netica.SetNodeInputName_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(link_index), 
                                   ctypes.c_char_p(name.encode()))
        checkerr()      
    
    def get_input_name(self, link_index):
        """Return the name of this node's input (i.e. link) identified by the given index.  
        
        Like Netica-C GetNodeInputName_bn.
        """
        Netica.GetNodeInputName_bn.restype = ctypes.c_char_p
        inputname = Netica.GetNodeInputName_bn(ctypes.c_void_p(self.cptr), 
                                                     ctypes.c_int(link_index)).decode()
        checkerr()
        return inputname
    
    def get_input_named(self, name):
        """Return the link index number of the link whose name is name.
        
        Returns -1 if there isn't one with that name (case sensitive comparison).
        Like Netica-C GetInputNamed_bn.
        """
        Netica.GetInputNamed_bn.restype = ctypes.c_int
        inputindex = Netica.GetInputNamed_bn(ctypes.c_char_p(name.encode()),
                                             ctypes.c_void_p(self.cptr))
        checkerr()
        return inputindex
 
    def set_input_delay(self, link_index, dimension, delay):
        """Set the time delay for the given link of this node in a dynamic Bayes net.  
        
        For link_index, can pass the name or the index of the link.
        Like Netica-C SetNodeInputDelay_bn.
        """
        if isinstance(link_index, str):
            link_index = self.get_input_named(link_index)
        Netica.SetNodeInputDelay_bn.restype = None
        Netica.SetNodeInputDelay_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(link_index),
                                    ctypes.c_int(dimension), ctypes.c_char_p(delay.encode()))
        checkerr()
  
    def set_persistence(self, dimension, persistence):
        """Set the amount of time that the value of this node can be considered 
        the same, before it must be recalculated in the next time slice of a 
        dynamic Bayes net.  
        
        Like Netica-C SetNodePersistence_bn.
        """
        Netica.SetNodePersistence_bn.restype = None
        Netica.SetNodePersistence_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(dimension),
                                     ctypes.c_char_p(persistence.encode()))
        checkerr()

    @property
    def equation(self):
        """Equation giving the probability of this node conditioned on its parent nodes, 
        or the value of this node as a function of its parents.  
        
        When setting, pass None for eqn to remove equation.
        Like Netica-C G/SetNodeEquation_bn.
        """
        Netica.GetNodeEquation_bn.restype = ctypes.c_char_p
        eqn = Netica.GetNodeEquation_bn(ctypes.c_void_p(self.cptr)).decode()
        checkerr()
        return eqn
    
    @equation.setter
    def equation(self, eqn):
        """Equation giving the probability of this node conditioned on its parent nodes, 
        or the value of this node as a function of its parents.  
        
        When setting, pass None for eqn to remove equation.
        Like Netica-C G/SetNodeEquation_bn.
        """
        if eqn is not None:
            eqn = ctypes.c_char_p(eqn.encode())
        Netica.SetNodeEquation_bn.restype = None
        Netica.SetNodeEquation_bn(ctypes.c_void_p(self.cptr), eqn)
        checkerr()
    
    def equation_to_table(self, num_samples, samp_unc, add_exist):
        """Build this node's CPT based on its equation.  
        
        See also the equation attribute.  
        Like Netica-C EquationToTable_bn.
        """
        Netica.EquationToTable_bn.restype = None
        Netica.EquationToTable_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(num_samples), 
                                  ctypes.c_ubyte(samp_unc), ctypes.c_ubyte(add_exist))
        checkerr() 
        
    def delete_tables(self):
        """Delete the CPT, experience, and function tables for this node, if it has any.
        
        Like Netica-C DeleteNodeTables_bn.
        """
        Netica.DeleteNodeTables_bn.restype = None
        Netica.DeleteNodeTables_bn(ctypes.c_void_p(self.cptr))
        checkerr()

    def has_table(self, complete=False):
        """Return True if node has a function table or a CPT table, otherwise False.
        
        It ignores experience tables.
        If complete is non-None, it is set to indicate whether node has a 
        complete table, i.e., none of the entries are undefined.
        Like Netica-C HasNodeTable_bn.
        """
        Netica.HasNodeTable_bn.retype = ctypes.c_bool
        has_table = Netica.HasNodeTable_bn(ctypes.c_void_p(self.cptr), 
                                           ctypes.c_bool(complete))
        checkerr()
        return bool(has_table)

    def reverse_link(self, parent):                 # May want to rename to reverse_link_from
        """Reverse the direction of the link from 'parent', keeping the overall 
        joint probabilities of the net unchanged.  
        
        For 'parent', you may pass the input name, input index, parent BNode or 
        parent name.  
        Like Netica-C ReverseLink_bn.
        """    
        if isinstance(parent, str):
            parent_node = self.net.get_node_named(parent)
            if parent_node:
                index = self.parents.index_of(parent_node)                
            if parent_node is None or index is None:
                index = self.get_input_named(parent)
                if index is -1:
                    errmesg = "Netica Error 6503: Function Node.reverse_link \
expects either the input name, input index, parent Node or parent \
name, but received a string that is not the name of any parent node or input"
                    raise NeticaPyError(errmesg)
                
            link_index = ctypes.c_int(index)
        elif isinstance(parent, Node):
            link_index = ctypes.c_int(self.parents.index_of(parent))
        elif isinstance(parent, int):
            link_index = ctypes.c_int(parent)
        else:
            raise TypeError('A Node, str, or int is required (got type {})'.format(type(parent).__name__))
        Netica.ReverseLink_bn.restype = None
        Netica.ReverseLink_bn(link_index, ctypes.c_void_p(self.cptr))
        checkerr()  

    def is_deterministic(self):
        """Whether the value of this node given its parents is deterministic 
        (versus probabilistic).  
        
        Like Netica-C IsNodeDeterministic_bn.
        """
        Netica.IsNodeDeterministic_bn.retype = ctypes.c_bool
        is_deterministic = Netica.IsNodeDeterministic_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return bool(is_deterministic)
    
    def set_user_field(self, name, data, kind=0):
        """Attach user defined data to this net under category 'field_name'.  
        
        Like Netica-C SetNodeUserField_bn.
        """
        if isinstance(data, str):
            data = data.encode()
        if isinstance(data, int):
            data = str(data).encode()
        length = len(data)
        Netica.SetNodeUserField_bn.restype = None
        Netica.SetNodeUserField_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(name.encode()),
                                  ctypes.c_char_p(data), ctypes.c_int(length), ctypes.c_int(kind))
        checkerr()      

    def get_user_field(self, name, return_type=None, kind=0):
        """Return user defined data to this net under category 'field_name'. 
        
        return_type can be set to 'str' or 'int' to convert the user field from
        a bytearray to a string or integer.
        Like Netica-C GetNodeUserField_bn.
        """
        clength = ctypes.c_int(0)
        Netica.GetNodeUserField_bn.restype = ctypes.POINTER(ctypes.c_ubyte)
        data_pointer = Netica.GetNodeUserField_bn(ctypes.c_void_p(self.cptr), ctypes.c_char_p(name.encode()),
                                         ctypes.byref(clength), ctypes.c_int(kind))
        checkerr()
        length = clength.value
        data = bytearray()
        
        for i in range(length):
            data.append(data_pointer[i])
        
        if return_type is 'str':
            data = data.decode()
        if return_type is 'int':
            data = int(data)
        
        return data

    def get_nth_user_field(self, index, kind=0):
        """Return the Nth element of user defined data from this net, and its 
        category 'field_name'.  
        
        Like Netica-C GetNodeNthUserField_bn.
        """
        clength = ctypes.c_int(0)
        cname = ctypes.c_char_p(b'')
        cvalue = ctypes.pointer(ctypes.c_ubyte(0))
        Netica.GetNodeNthUserField_bn.restype = None
        Netica.GetNodeNthUserField_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(index),
                                      ctypes.byref(cname), ctypes.byref(cvalue),
                                      ctypes.byref(clength), ctypes.c_int(kind))
        checkerr()
        length = clength.value
        name = cname.value
        data = bytearray()
        
        for i in range(length):
            data.append(cvalue[i])
        
        return name, data

    def set_vis_position(self, x, y, vis=None):
        """Set x, y to the coordinates of the center of node.
        

        This is useful when directly programming Netica Application, or before 
        writing a net to a file that will later be read by Netica Application.
        The min x is 84.0 pixels and the min y is 33.0 pixels.
        Like Netica-C SetNodeVisPosition_bn.
        """
        Netica.SetNodeVisPosition_bn.restype = None
        Netica.SetNodeVisPosition_bn(ctypes.c_void_p(self.cptr), vis, 
                                     ctypes.c_double(x), ctypes.c_double(y))
        checkerr()

    def get_vis_position(self, vis=None):
        """Return x, y coordinates of the center of node.
        
        Set as it would appear in a visual display, e.g., in Netica Application.
        The min x is 84.0 pixels and the min y is 33.0 pixels.
        Like Netica-C GetNodeVisPosition_bn.
        """
        cx = ctypes.c_double(0)
        cy = ctypes.c_double(0)
        Netica.GetNodeVisPosition_bn.restype = None
        Netica.GetNodeVisPosition_bn(ctypes.c_void_p(self.cptr), vis, 
                                     ctypes.byref(cx), ctypes.byref(cy))
        checkerr()
        return cx.value, cy.value

    def set_vis_style(self, style, vis=None):
        """The style to draw the node in the net diagram.
        
        style must be one of: "Default", "Absent", "Shape", "LabeledBox", 
        "BeliefBars", "BeliefLine", or "Meter" 
        Like Netica-C SetNodeVisStyle_bn.
        """
        Netica.SetNodeVisStyle_bn.retype = None
        Netica.SetNodeVisStyle_bn(ctypes.c_void_p(self.cptr), vis, 
                                  ctypes.c_char_p(style.encode()))
        checkerr()

    def get_vis_style(self, vis=None):
        """Return the current "style descriptor" of node for any visual display, 
        e.g., in Netica Application.

        The returned string may be used as a parameter to SetNodeVisStyle_bn.
        Like Netica-C GetNodeVisStyle_bn.
        """
        Netica.GetNodeVisStyle_bn.restype = ctypes.c_char_p
        style = Netica.GetNodeVisStyle_bn(ctypes.c_void_p(self.cptr), vis)
        checkerr()
        return style.decode()
    
    def add_link_from(self, parent):
        """Add a link from 'parent' node to this node.
        
        Like Netica-C AddLink_bn.
        """
        if not isinstance(parent, Node):
            raise TypeError('A Node is required (got type {})'.format(type(parent).__name__))
        
        Netica.AddLink_bn.restype = ctypes.c_int
        Netica.AddLink_bn(ctypes.c_void_p(parent.cptr), ctypes.c_void_p(self.cptr))
        checkerr()
   
    def delete_link_from(self, parent):
        """Delete the link entering this node from 'parent' node.  
        
        For 'parent' you may pass the input name, input index, parent Node or 
        parent name.  
        Like Netica-C DeleteLink_bn.
        """
        if isinstance(parent, str):
            parent_node = self.net.get_node_named(parent)
            if parent_node:
                index = self.parents.index_of(parent_node)                
            if parent_node is None or index is None:
                index = self.get_input_named(parent)
                if index is -1:
                    errmesg = "Netica Error 6501: Function Node.delete_link_from \
expects either the input name, input index, parent Node or parent \
name, but received a string that is not the name of any parent node or input"
                    raise NeticaPyError(errmesg)
                
            link_index = ctypes.c_int(index)
        elif isinstance(parent, Node):
            link_index = ctypes.c_int(self.parents.index_of(parent))
        elif isinstance(parent, int):
            link_index = ctypes.c_int(parent)
        else:
            raise TypeError('A Node, str, or int is required (got type {})'.format(type(parent).__name__))
        Netica.DeleteLink_bn.restype = None
        Netica.DeleteLink_bn(link_index, ctypes.c_void_p(self.cptr))
        checkerr()
     
    def switch_parent(self, parent, new_parent):
        """Switch the parent of the identified link with 'new_parent' node.  
        
        For 'parent' you may pass the input name, input index, parent BNode or 
        parent name, and for 'new_parent', you may pass its name or BNode.  
        Like Netica-C SwitchNodeParent_bn.
        """
        if isinstance(parent, str):
            try:
                parent_node = self.net.get_node_named(parent)
                index = self.parents.index_of(parent_node)
            except:
                index = self.get_input_named(parent)
                if index is -1:
                    errmesg = "Netica Error 6502: Function Node.switch_parent expects either the input name, \
input index, parent Node or parent name, but received a string that is not the name of any parent node or input"
                    raise NeticaPyError(errmesg)
                
            int_parent = ctypes.c_int(index)
        elif isinstance(parent, int):
            int_parent = ctypes.c_int(parent)
        else:
            int_parent = self.parents.index_of(parent)
            int_parent = ctypes.c_int(int_parent)
        
        if isinstance(new_parent, str):
            new_parent = self.net.get_node_named(new_parent).cptr
        elif new_parent is not None:
            if not isinstance(new_parent, Node):
                raise TypeError('A Node or string is required (got type {})'.format(type(new_parent).__name__))
            new_parent = new_parent.cptr
            
        Netica.SwitchNodeParent_bn.restype = None
        Netica.SwitchNodeParent_bn(int_parent, ctypes.c_void_p(self.cptr), 
                                   ctypes.c_void_p(new_parent))
        checkerr()
             
    def is_related(self, relation, node):
        """Return whether this node is related to 'node' by 'relation'.  
        
        For example Node.is_node_related("parent", other_node) returns whether 
        this node is a parent of other_node.  relation should be one of 
        "parent", "child", "ancestor", "descendent", "connected", 
        "markov_blanket", "d_connected".
        Like Netica-C IsNodeRelated_bn.
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        Netica.IsNodeRelated_bn.restype = ctypes.c_bool
        is_related = Netica.IsNodeRelated_bn(ctypes.c_void_p(self.cptr),
                                             ctypes.c_char_p(relation.encode()),
                                             ctypes.c_void_p(node.cptr))
        return bool(is_related)
     
    def get_related_nodes(self, nodelist, relation):
        """Put in 'related_nodes' those nodes that bear 'relation' to this node.  
        
        See also Net.get_related_nodes.  
        Like Netica-C GetRelatedNodes_bn.
        """
        if not isinstance(nodelist, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodelist).__name__))
        Netica.GetRelatedNodes_bn.restype = None
        Netica.GetRelatedNodes_bn(ctypes.c_void_p(nodelist.cptr), 
                                  ctypes.c_char_p(relation.encode()),
                                  ctypes.c_void_p(self.cptr))
        checkerr()    
    
    def add_states(self, first_state, state_names, num_states, cpt_fill=-1):
        """Add 'num_states' states, with the first to go after 'first_state'.  
        
        'state_names' can be empty, or if the existing states have names, it 
        can be a comma-delimited list of names for the new states.  
        Like Netica-C AddNodeStates_bn.
        """
        if state_names is not None:
            state_names = ctypes.c_char_p(state_names.encode())
        Netica.AddNodeStates_bn.restype = None
        Netica.AddNodeStates_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(first_state), 
                                state_names, ctypes.c_int(num_states), ctypes.c_double(cpt_fill))
        checkerr()
   
    def remove_state(self, state):
        """Remove 'state' from this node.  
        
        For 'state' pass its index or name.  
        Like Netica-C RemoveNodeState_bn.
        """
        if isinstance(state, str):
            state_num = self.get_state_named(state)
            state = ctypes.c_int(state_num)
        elif isinstance(state, int):
            state = ctypes.c_int(state)
        else:
            raise TypeError('A string or int is required (got type {})'.format(type(state).__name__))
        Netica.RemoveNodeState_bn.restype = None
        Netica.RemoveNodeState_bn(ctypes.c_char_p(self.cptr), state)
        checkerr()

    def reorder_states(self, new_order):
        """Put this node's states in the order given by 'new_order', which must 
        be an array of the existing states, indexes or names.
        
        Like Netica-C ReorderNodeStates2_bn.
        """
        length = len(new_order)
        cnew_order = (ctypes.c_int*length)(*new_order)
        Netica.ReorderNodeStates2_bn.restype = None
        Netica.ReorderNodeStates2_bn(ctypes.c_void_p(self.cptr), cnew_order, 
                                     ctypes.c_int(length))
        checkerr()

    def enter_finding(self, state):
        """Enter a finding for this node as an int, boolean or string.  
        
        Like Netica-C EnterFinding_bn.
        """
        Netica.EnterFinding_bn.restype = None
        if isinstance(state, str):
            state = self.get_state_named(state)
        Netica.EnterFinding_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state))
        checkerr()
    
    def enter_finding_not(self, state):
        """Enter a negative finding for this node as an int boolean or string.
        
        Finding entered indicates that it's not in the given state.  
        Like Netica-C EnterFindingNot_bn.
        """
        Netica.EnterFindingNot_bn.restype = None
        if isinstance(state, str):
            state = self.get_state_named(state)
        Netica.EnterFindingNot_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(state))
        checkerr()  
    
    def enter_value(self, value):
        """Enters a numerical real-valued finding for this node.  
            
        Like Netica-C EnterNodeValue_bn.
        """
        Netica.EnterNodeValue_bn.restype = None
        Netica.EnterNodeValue_bn(ctypes.c_void_p(self.cptr), ctypes.c_double(value))
        checkerr()             

    def enter_likelihood(self, likelihood):
        """Enter a likelihood vector for this node.  
        
        It is an array of one probability per state that need not sum to 1.  
        Like Netica-C EnterNodeLikelihood2_bn.
        """
        length = len(likelihood)
        clikelihood = (ctypes.c_float*length)(*likelihood)
        Netica.EnterNodeLikelihood2_bn.restype = None
        Netica.EnterNodeLikelihood2_bn(ctypes.c_void_p(self.cptr), 
                                       clikelihood, ctypes.c_int(length))
        checkerr()

    def enter_calibration(self, calibration):
        """For use by Norsys only.  
        
        Pass an array of one probability per state that sum to 1.  
        Like Netica-C EnterNodeCalibration_bn.
        """
        ccalibration = (ctypes.c_float*len(calibration))(*calibration)
        Netica.EnterNodeCalibration_bn.restype = None
        Netica.EnterNodeCalibration_bn(ctypes.c_void_p(self.cptr), ccalibration,
                                       ctypes.c_int(len(calibration)))
        checkerr()

    def enter_interval_finding(self, low, high):
        """Enter the finding that the value of this node is somewhere between low and high.  
        
        Like Netica-C EnterIntervalFinding_bn.
        """
        Netica.EnterIntervalFinding_bn.restype = None
        Netica.EnterIntervalFinding_bn(ctypes.c_void_p(self.cptr), ctypes.c_double(low),
                                       ctypes.c_double(high))
        checkerr()

    def enter_gaussian_finding(self, mean, std_dev):
        """Enter a Gaussian, i.e. 'normal', distributed likelihood finding.  
        
        Like Netica-C EnterGaussianFinding_bn.
        """
        Netica.EnterGaussianFinding_bn.restype = None
        Netica.EnterGaussianFinding_bn(ctypes.c_void_p(self.cptr), ctypes.c_double(mean),
                                       ctypes.c_double(std_dev))
        checkerr()    
     
    def get_finding(self):
        """Return the state finding entered for this node.
        
        Returns a 'SpecialFinding' code if another kind of finding is entered.  
        Like Netica-C GetNodeFinding_bn.
        """
        Netica.GetNodeFinding_bn.restype = ctypes.c_int
        finding = Netica.GetNodeFinding_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if finding < 0:
            finding = Finding(finding).name
        return finding
   
    def get_value_entered(self):
        """Return the real value finding entered for this node.  
        
        Like Netica-C GetNodeValueEntered_bn.
        """
        Netica.GetNodeValueEntered_bn.restype = ctypes.c_double
        val_entered = Netica.GetNodeValueEntered_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return val_entered
    
    def get_likelihood(self):
        """Return the accumulated (likelihood and other) findings for this node 
        as a likelihood for 'state'.  
        
        Like Netica-C GetNodeLikelihood2_bn.
        """
        clength = ctypes.c_int(0)
        Netica.GetNodeLikelihood2_bn.restype = ctypes.POINTER(ctypes.c_float)
        likelihood_pointer = Netica.GetNodeLikelihood2_bn(ctypes.c_void_p(self.cptr),
                                                          ctypes.byref(clength))
        checkerr()     
        length = clength.value
        if likelihood_pointer:
            likelihood = []
            for i in range(length):
                likelihood.extend([likelihood_pointer[i]]) 
        else:
            likelihood = None
        return likelihood
    
    def retract_findings(self):
        """Retract findings previously entered for this node. 
        
        Retract state, real-valued and likelihood findings.  
        Like Netica-C RetractNodeFindings_bn.
        """
        Netica.RetractNodeFindings_bn.restype = None
        Netica.RetractNodeFindings_bn(ctypes.c_void_p(self.cptr))
        checkerr()     
   
    def calc_state(self):
        """Gets this node's state calculated from neighboring nodes, if their 
        values can be found, and the relationship with those nodes will allow 
        it, e.g. is deterministic.  
        
        See also calc_value, get_beliefs.  
        Like Netica-C CalcNodeState_bn.
        """
        Netica.CalcNodeState.restype = ctypes.c_int
        state = Netica.CalcNodeState(ctypes.c_void_p(self.cptr))
        checkerr()
        return state
     
    def calc_value(self):
        """Gets this node's value calculated from neighboring nodes, if their 
        values can be found, and the relationship with those nodes will allow 
        it, e.g. is deterministic.  
        
        See also calc_state.  
        Like Netica-C CalcNodeValue_bn.
        """
        Netica.CalcNodeValue_bn.restype = ctypes.c_double
        value = Netica.CalcNodeValue_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return value

    def is_belief_updated(self):
        """Return whether the beliefs for this node have already been 
        calculated by belief propagation.  
        
        Like Netica-C IsBeliefUpdated_bn.
        """
        Netica.IsBeliefUpdated_bn.retype = ctypes.c_bool
        is_updated = Netica.IsBeliefUpdated_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return bool(is_updated) 
        
    def get_beliefs(self):
        """Return a belief vector indicating the current probability for each state of node.
        
        Gets the current belief for each state of this nature node, taking into 
        account all findings entered in the net. See also CalcState, 
        GetExpectedValue, GetExpectedUtility.
        Like Netica-C GetNodeBeliefs2_bn.
        """ 
        clength = ctypes.c_int(0)
        Netica.GetNodeBeliefs2_bn.restype = ctypes.POINTER(ctypes.c_float)
        beliefs_pointer = Netica.GetNodeBeliefs2_bn(ctypes.c_void_p(self.cptr),
                                                    ctypes.byref(clength))
        checkerr()       
        length = clength.value
        if beliefs_pointer:
            beliefs = []
            for i in range(length):
                beliefs.extend([beliefs_pointer[i]])  
        else:
            beliefs = None
        return beliefs 
    
    def get_expected_value(self, num_moments=1):
        """Return the expected value for this real valued node.
        
        Passing higher values for num_moments (to a maximum of 4) will produce
        a tuple containing the moments of the funtion. For example, passing 2 
        will return the expected value and the standard deviation. Results are
        based on the current beliefsof the node (i.e. taking into account all 
        findings entered in the net).  
        
        See also get_beliefs, get_expected_utils.  
        Like Netica-C GetNodeExpectedValue_bn.
        """
        if num_moments < 1 or num_moments > 4:
            raise ValueError('num_moments must be between 1 and 4 inclusive')
        if num_moments > 1:
            std_dev = ctypes.c_double(0)
            std_dev_ref = ctypes.pointer(std_dev)
        else:
            std_dev_ref = None
        if num_moments > 2:
            x3 = ctypes.c_double(0)
            x3_ref = ctypes.pointer(x3)
        else:
            x3_ref = None
        if num_moments > 3:
            x4 = ctypes.c_double(0)
            x4_ref = ctypes.pointer(x4)
        else:
            x4_ref = None
        Netica.GetNodeExpectedValue_bn.restype = ctypes.c_double
        mean = Netica.GetNodeExpectedValue_bn(ctypes.c_void_p(self.cptr),
                                              std_dev_ref, x3_ref, x4_ref)
        checkerr()
        
        if num_moments == 1:
            return mean
        if num_moments > 1:
            return mean, std_dev.value
        if num_moments > 2:
            return mean, std_dev.value, x3.value
        if num_moments > 3:
            return mean, std_dev.value, x3.value, x4.value
    
    def get_expected_utils(self):
        """Get the current expected utility for each choice of this decision node. 
        
        Takes into account all findings entered in the net.     
        See also get_beliefs, get_expected_value.  
        Like Netica-C GetNodeExpectedUtils2_bn.
        """
        clength = ctypes.c_int(0)
        Netica.GetNodeExpectedUtils2_bn.restype = ctypes.POINTER(ctypes.c_float)
        expectedutils_pointer = Netica.GetNodeExpectedUtils2_bn(ctypes.c_void_p(self.cptr),
                                                                ctypes.byref(clength))
        checkerr()       
        length = clength.value
        if expectedutils_pointer:
            expectedutils = []
            for i in range(length):
                expectedutils.extend([expectedutils_pointer[i]])
        else:
            expectedutils = None
        return expectedutils

    def set_func_state(self, parentstates, state):
        """Set the table entry giving the state of this node for the row 'parentstates'. 
        
        Pass a list of state indexes or a comma-delimited string of state names.
        For continuous nodes, normally use RealFuncTable instead.  
        Like Netica-C SetNodeFuncState_bn
        """
        if isinstance(parentstates, list):
            numparentstates = parentstates
        elif isinstance(parentstates, str):
            numparentstates = []
            for statename in parentstates.split(', '):
                statenum = self.get_state_named(statename)              
                numparentstates.append(statenum)
        else:
            raise TypeError('A list or string is required (got type {})'.format(type(parentstates).__name__))
            
        cparentstates = (ctypes.c_int*len(numparentstates))(*numparentstates)
        Netica.SetNodeFuncState_bn.restype = None
        Netica.SetNodeFuncState_bn(ctypes.c_void_p(self.cptr), cparentstates, 
                                   ctypes.c_int(state))
        checkerr() 
    
    def get_func_state(self, parentstates):
        """Return the table entry giving the state of this node for the row 'parentstates'. 
        
        Pass a Condition, array of state indexes or comma-delimited string of 
        state names.  
        For continuous nodes, normally use RealFuncTable instead.  
        Like Netica-C GetNodeFuncState_bn
        """
        cparentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        Netica.GetNodeFuncState_bn.restype = ctypes.c_int
        nodefuncstates = Netica.GetNodeFuncState_bn(ctypes.c_void_p(self.cptr), cparentstates)
        checkerr()
        return nodefuncstates
    
    def set_func_real(self, parentstates, val):
        """A table entry giving the real value of this node for the row 'parentstates' 
        
        (Pass a Condition, array of state indexes or comma-delimited string of state names). 
        For discrete nodes, normally use StateFuncTable instead. 
        Like Netica-C SetNodeFuncReal_bn.
        """   
        cparentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        Netica.SetNodeFuncReal_bn.restype = None
        Netica.SetNodeFuncReal_bn(ctypes.c_void_p(self.cptr), cparentstates, ctypes.c_double(val))
        checkerr()

    def get_func_real(self, parentstates):
        """A table entry giving the real value of this node for the row 'parentstates' 
        
        (Pass a Condition, array of state indexes or comma-delimited string of state names). 
        For discrete nodes, normally use StateFuncTable instead. 
        Like Netica-C GetNodeFuncReal_bn.
        """
        cparentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        Netica.GetNodeFuncState_bn.restype = ctypes.c_double
        nodefuncreal = Netica.GetNodeFuncState_bn(ctypes.c_void_p(self.cptr), cparentstates)
        checkerr()
        return nodefuncreal

    def set_CPT(self, parentstates, probs):
        """Set probabilities for the states of this node given the passed parentstates.
        
        As a Condition, array of state indexes or comma-delimited string of 
        state names in the same order as this node's parents.  
        Like Netica-C SetNodeProbs_bn.
        """
        if parentstates is None:
            cparentstates = parentstates
            
        elif isinstance(parentstates, list):
            cparentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        
        elif isinstance(parentstates, str):
            statenames = parentstates.split(', ')
            parents = self.parents
            numparents = parents.length
            parent_states = []
            for i in range(numparents):
                nthparent = parents.nth_node(i)             
                state = nthparent.get_state_named(statenames[i])
                parent_states.append(state)  
            cparentstates = (ctypes.c_int*len(parent_states))(*parent_states)
        
        else:
            raise TypeError('A string, list, or None is required (got type {})'.format(type(parentstates).__name__))
        
        cprobs = (ctypes.c_float*len(probs))(*probs)
        
        Netica.SetNodeProbs_bn.restype = None
        Netica.SetNodeProbs_bn(ctypes.c_void_p(self.cptr), cparentstates, cprobs)
        checkerr()

    def get_CPT(self, parentstates):
        """Get probabilities for the states of this node given the passed parentstates.
        
        As a Condition, array of state indexes or comma-delimited string of 
        state names in the same order as this node's parents.  
        Like Netica-C GetNodeProbs_bn.
        """
        if parentstates is None:
            cparentstates = parentstates
            
        elif isinstance(parentstates, list):
            cparentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        
        elif isinstance(parentstates, str):
            statenames = parentstates.split(', ')
            parents = self.parents
            numparents = parents.length
            parent_states = []
            for i in range(numparents):
                nthparent = parents.nth_node(i)             
                state = nthparent.get_state_named(statenames[i])
                parent_states.append(state)  
            cparentstates = (ctypes.c_int*len(parent_states))(*parent_states)
        
        else:
            raise TypeError('A string, list, or None is required (got type {})'.format(type(parentstates).__name__))
        
        Netica.GetNodeProbs_bn.restype = ctypes.POINTER(ctypes.c_float)
        CPT_pointer = Netica.GetNodeProbs_bn(ctypes.c_void_p(self.cptr), cparentstates)
        checkerr()
           
        size = 1
        for n in range(self.parents.length):
            num_states = self.parents.nth_node(n).num_states
            # Include checks for if num_states is 0 or too large
            size *= num_states

        if CPT_pointer:
            CPT = []
            for i in range(size):
                CPT.extend([CPT_pointer[i]]) 
        else:
            CPT = None
        return CPT

    def set_experience(self, parentstates, experience):
        """Set the experience of this node as a function of its parent nodes.  
        
        For 'parent_states', pass a comma-delimited string listing them in the 
        same order as parents, or an array of their state indexes.
        Like Netica-C SetNodeExperience_bn.
        """
        if parentstates is None:
            cparentstates = parentstates
            
        elif isinstance(parentstates, list):
            cparentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        
        elif isinstance(parentstates, str):
            statenames = parentstates.split(', ')
            parents = self.parents
            numparents = parents.length
            parent_states = []
            for i in range(numparents):
                nthparent = parents.nth_node(i)             
                state = nthparent.get_state_named(statenames[i])
                parent_states.append(state)  
            cparentstates = (ctypes.c_int*len(parent_states))(*parent_states)
        
        else:
            raise TypeError('A string, list, or None is required (got type {})'.format(type(parentstates).__name__))
        
        Netica.SetNodeExperience_bn.restype = None
        Netica.SetNodeExperience_bn(ctypes.c_void_p(self.cptr), cparentstates,
                                    ctypes.c_double(experience))
        checkerr()
    
    def get_experience(self, parentstates):
        """Return the "experience" of the node for the situation described by the parent states.
        
        The order of the states in parentstates should match the order of the 
        nodes in the list returned by Node.parents.
        Like Netica-C GetNodeExperience_bn.
        """
        if parentstates is not None:                                          
            parentstates = (ctypes.c_int*len(parentstates))(*parentstates)
        Netica.GetNodeExperience_bn.restype = ctypes.c_double
        experience = Netica.GetNodeExperience_bn(ctypes.c_void_p(self.cptr), parentstates)
        checkerr()
        return experience
   
    def new_sensitivity(self, findings_nodes, what_calc):
        """Create a sensitivity measurer to determine how much this node could 
        be affected by new findings at certain other nodes. 
        
        Like Netica-C NewSensvToFinding_bn.
        """
        if not isinstance(findings_nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(findings_nodes).__name__))
        Netica.NewSensvToFinding_bn.restype = ctypes.c_void_p
        cptr = Netica.NewSensvToFinding_bn(ctypes.c_void_p(self.cptr), 
                                           ctypes.c_void_p(findings_nodes.cptr),
                                           ctypes.c_int(what_calc.value))
        checkerr()
        return Sensitivity(cptr, self)

    def add_to_nodeset(self, nodeset):
        """Add this node to the node-set named nodeset.
        
        Creates a new node-set if nodeset is not yet present in the net 
        containing node.
        Like Netica-C AddNodeToNodeset_bn.
        """
        Netica.AddNodeToNodeset_bn.restype = None
        Netica.AddNodeToNodeset_bn(ctypes.c_void_p(self.cptr), 
                                   ctypes.c_char_p(nodeset.encode()))
        checkerr()
    
    def remove_from_nodeset(self, nodeset):
        """Remove this node from the node-set named nodeset.
        
        Like Netica-C RemoveNodeFromNodeset_bn.
        """
        Netica.RemoveNodeFromNodeset_bn.restype = None
        Netica.RemoveNodeFromNodeset_bn(ctypes.c_void_p(self.cptr), 
                                        ctypes.c_char_p(nodeset.encode()))
        checkerr()

    def is_in_nodeset(self, nodeset):
        """Return whether this node is a member of nodeset.
        
        Like Netica-C IsNodeInNodeset_bn.
        """
        Netica.IsNodeInNodeset_bn.restype = ctypes.c_bool
        in_nodeset = Netica.IsNodeInNodeset_bn(ctypes.c_void_p(self.cptr),
                                               ctypes.c_char_p(nodeset.encode()))
        checkerr()
        return bool(in_nodeset)

    def make_probs_uniform(self):
        """Gives the passed node a uniform conditional probability distribution,
        i.e. all the probabilities the same.
        
        Like Netica-Ex 
        """
        num_states = self.num_states
        num_parents = self.parents.length
        uniform = []
        pstates = []
        
        for st in range(num_states):
            uniform.extend([1.0/num_states])
            
        for pn in range(num_parents):
            pstates.extend([-5])  #  EVERY_STATE = -5

        c_pstates = (ctypes.c_int*len(pstates))(*pstates)
        c_uniform = (ctypes.c_float*len(uniform))(*uniform)
      
        Netica.SetNodeProbs_bn.restype = None
        Netica.SetNodeProbs_bn(ctypes.c_void_p(self.cptr), c_pstates, c_uniform)
        
        checkerr()


"""------------------------------NodeList Class-----------------------------"""         
    
class NodeList:
    
    def __init__(self, cptr, is_const):
        
        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
            
        if is_const:
            self.is_const = True
        else:
            self.is_const = False
        
        
    def __del__(self):
        """Remove this node list, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteNodeList_bn.
        """
        if not self.is_const:
            Netica.DeleteNodeList_bn.restype = None
            Netica.DeleteNodeList_bn(ctypes.c_void_p(self.cptr))
            checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
            
    '''    
    def delete(self):
        """Remove this node list, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteNodeList_bn.
        """
        Netica.DeleteNodeList_bn.restype = None
        Netica.DeleteNodeList_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''
    
    def clear(self):
        """Remove all entries from this node list, so it becomes empty.  
        
        Like Netica-C ClearNodeList_bn.
        """
        Netica.ClearNodeList_bn.restype = None
        Netica.ClearNodeList_bn(ctypes.c_void_p(self.cptr))
    
    @property
    def length(self):
        """Return the number of nodes and empty entries in this list.  
        
        Like Netica-C LengthNodeList_bn.
        """    
        Netica.LengthNodeList_bn.restype = ctypes.c_int
        length = Netica.LengthNodeList_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        return length

    def add(self, node, index):
        """Insert the given node at the given index.  
        
        Passing None for index adds it to the end.  
        Like Netica-C AddNodeToList_bn.
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        if index is None:
            index = -10
        Netica.AddNodeToList_bn.restype = None
        Netica.AddNodeToList_bn(ctypes.c_void_p(node.cptr), ctypes.c_void_p(self.cptr), 
                                ctypes.c_int(index))
        checkerr()
        
    def remove(self, index):
        """Remove (and return) the node at the position indicated by 'index'.
        
        Index can be an integer, BNode, or node name. 
        Passing 'LAST_ENTRY' for index removes it from the end. 
        Like Netica-C RemoveNthNode_bn.
        """
        if isinstance(index, str):
            if index is 'LAST_ENTRY':
                index = -10
            else:
                net = self.nth_node(0).net
                node = net.get_node_named(index)
                index = self.index_of(node)
        elif isinstance(index, Node):
            index = self.index_of(index)

        Netica.RemoveNthNode_bn.restype = ctypes.c_void_p
        cptr = Netica.RemoveNthNode_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(index))
        checkerr()
        
        return create_node(cptr)
        
    def nth_node(self, index):                              # May want to rename to get_nth_node
        """Return the nth node in this list (first is 0).  
        
        Can also access by name string instead of integer index. 
        Like Netica-C NthNode_bn.
        """
        if isinstance(index, str):
            index = self.index_of(index)
        Netica.NthNode_bn.restype = ctypes.c_void_p
        cptr = Netica.NthNode_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(index))
        checkerr()
        
        return create_node(cptr)
    
    def set_nth_node(self, node, index):  
        """Put node at position index of list nodes without changing the length of the list.
        
        Can also access by name string instead of integer index.
        Like Netica-C SetNthNode_bn.
        """
        if isinstance(index, str):
            index = self.index_of(index)
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))    
        Netica.SetNthNode_bn.restype = None
        Netica.SetNthNode_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(index), 
                             ctypes.c_void_p(node.cptr))
        checkerr()

    def index_of(self, node, start_index=0):
        """Return the position (index) of 'node' in this list  
        
        Return the index of the first occuence of 'node', or None if it is not 
        found. If 'start_index' is passed, the search starts on that index and 
        goes to the end.  
        Like Netica-C IndexOfNodeInList_bn.
        """
        if isinstance(node, str):
            net = self.nth_node(0).net
            node = net.get_node_named(node)
            if node is None:
                errmesg = "Netica Error 6504: Function NodeList.index_of received \
                a string that was not the name of any node in the Net."
                raise NeticaPyError(errmesg)
            
        if not isinstance(node, Node):
            raise TypeError('A Node or str is required (got type {})'.format(type(node).__name__))
        
        Netica.IndexOfNodeInList_bn.restype = ctypes.c_int
        index = Netica.IndexOfNodeInList_bn(ctypes.c_void_p(node.cptr), 
                                            ctypes.c_void_p(self.cptr), ctypes.c_int(start_index))
        checkerr()
        if index == -1:
            return None
        else:
            return index
    
    def copy(self):
        """Duplicate the list nodes, and return the duplicate list.
        
        This only makes a copy of the list; if you want to duplicate the nodes 
        as well, use Net.copy_nodes.
        Like Netica-C DupNodeList_bn.
        """            
        Netica.DupNodeList_bn.restype = ctypes.c_void_p
        cptr = Netica.DupNodeList_bn(ctypes.c_void_p(self.cptr)) 
        checkerr()
        
        return create_nodelist(cptr)

    @property
    def nodes(self):
        """Return a python list of all the nodes in this NodeList.
        """
        #PythonOnly
        num_nodes = self.length
        py_list = []
        for i in range(num_nodes):
            py_list.append(self.nth_node(i))       
        return py_list
    

"""------------------------------Caseset Class------------------------------"""

class Caseset:

    def __init__(self, cptr):

        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
    
    def __del__(self):
        """Remove this Caseset, freeing all resources it consumes, including memory.  
               
        Will not delete files this Caseset refers to.  
        Like Netica-C DeleteCaseset_cs.
        """
        Netica.DeleteCaseset_cs.restype = None
        Netica.DeleteCaseset_cs(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
    
    '''
    def delete(self):
        """Remove this Caseset, freeing all resources it consumes, including memory.  
               
        Will not delete files this Caseset refers to.  
        Like Netica-C DeleteCaseset_cs.
        """
        Netica.DeleteCaseset_cs.restype = None
        Netica.DeleteCaseset_cs(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''
        
    def add_cases_from_file(self, file, degree, options=None):
        """Add all the cases contained in 'file'.  
        
        'file' is the name of a stream
        'degree' (normally 1) is a multiplier for the multiplicty of each case.  
        'options' must be empty or None.  
        Like Netica-C AddFileToCaseset_cs.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
            
        if not isinstance(file, Stream):
            raise TypeError('A Stream is required (got type {})'.format(type(file).__name__))
            
        Netica.AddFileToCaseset_cs.restype = None
        Netica.AddFileToCaseset_cs(ctypes.c_void_p(self.cptr), ctypes.c_void_p(file.cptr),
                                   ctypes.c_double(degree), options)
        checkerr()
    
    def add_cases_from_database(self, dbmgr, degree, nodes, column_names, tables, condition, options):
        """Adds to this Caseset the cases contained in db.  
        
        Like Netica-C AddDBCasesToCaseset_cs.
        """
        if column_names is not None:
            column_names = ctypes.c_char_p(column_names.encode())
        if tables is not None:
            tables = ctypes.c_char_p(tables.encode())
        if condition is not None:
            condition = ctypes.c_char_p(condition.encode())
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        
        if not isinstance(dbmgr, DatabaseManager):
            raise TypeError('A DatabaseManager is required (got type {})'.format(type(dbmgr).__name__))
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        
        Netica.AddDBCasesToCaseset_cs.restype = None
        Netica.AddDBCasesToCaseset_cs(ctypes.c_void_p(self.cptr), ctypes.c_void_p(dbmgr.cptr),
                                      ctypes.c_double(degree), ctypes.c_void_p(nodes.cptr),
                                      column_names, tables, condition, options)
        checkerr()

    def write(self, file, options=None):
        """Write all the cases to the indicated file (which may be on disk or in memory). 
        
        'options' must be empty or NULL.  
        Like Netica-C WriteCaseset_cs.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
            
        if not isinstance(file, Stream):
            raise TypeError('A Stream is required (got type {})'.format(type(file).__name__))
            
        Netica.WriteCaseset_cs.restype = None
        Netica.WriteCaseset_cs(ctypes.c_void_p(self.cptr), 
                               ctypes.c_void_p(file.cptr), options)
        checkerr()

"""-----------------------------DatabaseManager Class-----------------------------"""

class DatabaseManager:
    
    def __init__(self, cptr):
        
        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)

    def __del__(self):
        """"Remove this DatabaseManager.
        
        Closes connections and frees all the resources it consumes, including 
        memory.  
        Like Netica-C DeleteDBManager_cs.
        """
        Netica.DeleteDBManager_cs.restype = None
        Netica.DeleteDBManager_cs(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        
    '''
    def delete(self):
        """"Remove this DatabaseManager.
        
        Closes connections and frees all the resources it consumes, including 
        memory.  
        Like Netica-C DeleteDBManager_cs.
        """
        Netica.DeleteDBManager_cs.restype = None
        Netica.DeleteDBManager_cs(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''

    def execute_sql(self, sql_cmd, options=None):
        """Execute the passed SQL statement.  
        
        Like Netica-C ExecuteDBSql_cs.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.ExecuteDBSql_cs.restype = None
        Netica.ExecuteDBSql_cs(ctypes.c_void_p(self.cptr), 
                               ctypes.c_char_p(sql_cmd.encode()), options)
        checkerr()
        
    def insert_findings(self, nodes, column_names, tables, options=None):
        """Add a new record to the database, consisting of the findings in 'nodes'.  
        
        If 'column_names' non-empty, it contains the database names for each of the variables in 'nodes'.  
        'tables' empty means use default table.  
        Like Netica-C InsertFindingsIntoDB_bn.
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
            
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
            
        Netica.InsertFindingsIntoDB_bn.restype = None
        Netica.InsertFindingsIntoDB_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(nodes.cptr),
                                       ctypes.c_char_p(column_names.encode()),
                                       ctypes.c_char_p(tables.encode()), options)
        checkerr()
    
    def add_nodes(self, net, column_names, tables, condition, options):
        """***nodocs
        
        Like Netica-C AddNodesFromDB_bn.
        """
        if column_names is not None:
            column_names = ctypes.c_char_p(column_names.encode())
        if tables is not None:
            tables = ctypes.c_char_p(tables.encode())
        if condition is not None:
            condition = ctypes.c_char_p(condition.encode())
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        
        if not isinstance(net, Net):
            raise TypeError('A Net is required (got type {})'.format(type(net).__name__))
            
        Netica.AddNodesFromDB_bn.restype = None
        Netica.AddNodesFromDB_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(net.cptr),
                                 column_names, tables, condition, options)
        checkerr()

"""------------------------------Learner Class------------------------------"""  
      
class Learner:
    
    def __init__(self, cptr):
        
        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)

    def __del__(self):
        """Remove this Learner, freeing all resources it consumes, including memory. 
        
        Like Netica-C DeleteLearner_bn.
        """
        Netica.DeleteLearner_bn.restype = None
        Netica.DeleteLearner_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None    
 
    '''
    def delete(self):
        """Remove this Learner, freeing all resources it consumes, including memory. 
        
        Like Netica-C DeleteLearner_bn.
        """
        Netica.DeleteLearner_bn.restype = None
        Netica.DeleteLearner_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''
      
    @property
    def max_iterations(self):
        """Parameter to control termination.  
        
        The maximum number of learning step iterations.  
        Like Netica-C SetLearnerMaxIters_bn.
        """
        Netica.SetLearnerMaxIters_bn.restype = ctypes.c_int
        maxiters = Netica.SetLearnerMaxIters_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(-1))
        checkerr()
        return maxiters
    
    @max_iterations.setter
    def max_iterations(self, max_iters):
        """Parameter to control termination.
        
        The maximum number of learning step iterations.  
        Like Netica-C SetLearnerMaxIters_bn.
        """
        Netica.SetLearnerMaxIters_bn.restype = ctypes.c_int
        Netica.SetLearnerMaxIters_bn(ctypes.c_void_p(self.cptr), ctypes.c_int(max_iters))
        checkerr()
    
    @property
    def max_tolerance(self):
        """Parameter to control termination.  
        
        When the log likelihood of the data given the model improves by less 
        than this, the model is considered close enough.  
        Like Netica-C SetLearnerMaxTol_bn.
        """
        Netica.SetLearnerMaxTol_bn.restype = ctypes.c_double
        maxtol = Netica.SetLearnerMaxTol_bn(ctypes.c_void_p(self.cptr), ctypes.c_double(QUERY_ns))
        checkerr()
        return maxtol
    
    @max_tolerance.setter
    def max_tolerance(self, max_tol):
        """Parameter to control termination.
        
        When the log likelihood of the data given the model improves by less 
        than this, the model is considered close enough.  
        Like Netica-C SetLearnerMaxTol_bn.
        """
        Netica.SetLearnerMaxTol_bn.restype = ctypes.c_double
        Netica.SetLearnerMaxTol_bn(ctypes.c_void_p(self.cptr), ctypes.c_double(max_tol))
        checkerr()
      
    def learn_CPTs(self, nodes, cases, degree):
        """Modify the CPTs (and experience tables) for the nodes in 'nodes', 
        to take into account the case data from 'cases' (with multiplicity multiplier 'degree', which is normally 1).  
        Like Netica-C LearnCPTs_bn.
        """
        if not isinstance(nodes, NodeList):
            raise TypeError('A NodeList is required (got type {})'.format(type(nodes).__name__))
        if not isinstance(cases, Caseset):
            raise TypeError('A Caseset is required (got type {})'.format(type(cases).__name__))
            
        Netica.LearnCPTs_bn.restype = None
        Netica.LearnCPTs_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(nodes.cptr),
                            ctypes.c_void_p(cases.cptr), ctypes.c_double(degree))
        checkerr()

"""------------------------------Tester Class-------------------------------"""  
    
class Tester:
    
    def __init__(self, cptr):
        
        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
   
    def __del__(self):
        """Remove this net tester, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteNetTester_bn.
        """
        Netica.DeleteNetTester_bn.restype = None
        Netica.DeleteNetTester_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
    
    '''
    def delete(self):
        """Remove this net tester, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteNetTester_bn.
        """
        Netica.DeleteNetTester_bn.restype = None
        Netica.DeleteNetTester_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''

    def test_with_cases(self, cases):
        """Scan through the data in 'cases', and for each case check the values 
        in the case for test_nodes against the predictions made by the net 
        based on the other values in the case.  
        
        test_nodes is set with BNet.NewNetTester.  
        Like Netica-C TestWithCaseset_bn
        """
        if not isinstance(cases, Caseset):
            raise TypeError('A Caseset is required (got type {})'.format(type(cases).__name__))
        Netica.TestWithCaseset_bn.restype = None
        Netica.TestWithCaseset_bn(ctypes.c_void_p(self.cptr), ctypes.c_void_p(cases.cptr))
        checkerr()

    def error_rate(self, node):
        """Return the fraction of test cases where the net predicted the wrong state.  
        
        Like Netica-C GetTestErrorRate_bn.
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        Netica.GetTestErrorRate_bn.restype = ctypes.c_double
        error_rate = Netica.GetTestErrorRate_bn(ctypes.c_void_p(self.cptr), 
                                                ctypes.c_void_p(node.cptr))
        checkerr()
        return error_rate

    def log_loss(self, node):
        """The 'logarithmic loss', which for each case takes into account the 
        prediction probability the net gives to the state that turns out to be correct.  
        
        Ranges from 0 (perfect score) to infinity.  
        Like Netica-C GetTestLogLoss_bn.
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        Netica.GetTestLogLoss_bn.restype = ctypes.c_double
        log_loss = Netica.GetTestLogLoss_bn(ctypes.c_void_p(self.cptr), 
                                            ctypes.c_void_p(node.cptr))
        checkerr()
        return log_loss

    def quadratic_loss(self, node):
        """The 'quadratic loss', also known as 'Brier score' for 'node' under 
        the test performed by TestWithCases.  
        
        Like Netica-C GetTestQuadraticLoss_bn.
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        Netica.GetTestQuadraticLoss_bn.restype = ctypes.c_double
        quadratic_loss = Netica.GetTestQuadraticLoss_bn(ctypes.c_void_p(self.cptr),
                                                        ctypes.c_void_p(node.cptr))
        checkerr()
        return quadratic_loss
        
    def get_confusion_matrix(self, node, predicted, actual):
        """Return an element of the 'confusion matrix'. 
        
        Element is the number of times the net predicted 'predicted_state' for 
        node, but the case file actually held 'actual_state' as the value of 
        that node.  
        Like Netica-C GetTestConfusion_bn.
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        if isinstance(predicted, str):
            predicted = node.get_state_named(predicted)
        if isinstance(actual, str):
            actual = node.get_state_named(actual)
        Netica.GetTestConfusion_bn.restype = ctypes.c_double
        test_confusion = Netica.GetTestConfusion_bn(ctypes.c_void_p(self.cptr), 
                                                    ctypes.c_void_p(node.cptr),
                                                    ctypes.c_int(predicted), 
                                                    ctypes.c_int(actual))
        checkerr()
        return test_confusion

    def binary_score(self, score, node, positive_state, granularity=-1):
        """***nodocs
        """
        if not isinstance(node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(node).__name__))
        cnum_entries = ctypes.c_int(0)
        thresholds_ref = ctypes.POINTER(ctypes.c_double)()
        Netica.GetTestBinaryScore_bn.restype = ctypes.POINTER(ctypes.c_double)
        score_pointer = Netica.GetTestBinaryScore_bn(ctypes.c_void_p(self.cptr), 
                                                     ctypes.c_char_p(score.encode()),
                                                     ctypes.c_void_p(node.cptr), 
                                                     ctypes.c_int(positive_state),
                                                     ctypes.c_double(granularity), 
                                                     ctypes.byref(cnum_entries),
                                                     ctypes.byref(thresholds_ref))
        checkerr()

        binary_score = score_pointer[0]        
        num_entries = cnum_entries.value
        
        thresholds = []
        for i in range(num_entries):
            thresholds.append(thresholds_ref[i])
        
        return binary_score, thresholds
        
        

"""--------------------------RandomGenerator Class--------------------------"""

class RandomGenerator:
    
    def __init__(self, cptr):

        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
        
    def __del__(self):
        """Remove this random generator, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteRandomGen_ns.
        """
        Netica.DeleteRandomGen_ns.restype = None
        Netica.DeleteRandomGen_ns(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None    
    
    '''    
    def delete(self):
        """Remove this random generator, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteRandomGen_ns.
        """
        Netica.DeleteRandomGen_ns.restype = None
        Netica.DeleteRandomGen_ns(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''
    
    def get_state(self, options=None):
        """***nodocs
        
        Like Netica-C GetRandomGenState_ns
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        Netica.GetRandomGenState_ns.restype = ctypes.c_char_p
        seed = Netica.GetRandomGenState_ns(ctypes.c_void_p(self.cptr), options)
        checkerr()
        return seed.decode()

    def generate_random_numbers(self, num, options=None):
        """***nodocs
        """
        if options is not None:
            options = ctypes.c_char_p(options.encode())
        #results = ctypes.c_double(0)
        #results_ref = ctypes.POINTER(results)
        array = [0] * num
        results = (ctypes.c_double*num)(*array)
        Netica.GenerateRandomNumbers_ns.restype = ctypes.c_double
        Netica.GenerateRandomNumbers_ns(ctypes.c_void_p(self.cptr), results, 
                                        ctypes.c_int(num), options)
        checkerr()    
        rand_nums = []
        for i in range(num):
            rand_nums.append(results[i])
        return rand_nums

    
"""----------------------------Sensitivity Class----------------------------"""

class Sensitivity:
    
    def __init__(self, cptr, t_node):

        self.cptr = cptr
        
        if dict_initialization:
            cptr_dict[cptr] = weakref.ref(self)
            
        self.node = t_node
        
    def __del__(self):
        """Remove this sensitivity measurer, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteSensvToFinding_bn.
        """
        Netica.DeleteSensvToFinding_bn.restype = None
        Netica.DeleteSensvToFinding_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None

    '''
    def delete(self):
        """Remove this sensitivity measurer, freeing all resources it consumes, including memory.  
        
        Like Netica-C DeleteSensvToFinding_bn.
        """
        Netica.DeleteSensvToFinding_bn.restype = None
        Netica.DeleteSensvToFinding_bn(ctypes.c_void_p(self.cptr))
        checkerr()
        if dict_initialization:
            del cptr_dict[self.cptr]
        self.cptr = None
        '''
    
    def get_mutual_info(self, finding_node):
        """Return the mutual information between q_node and finding_node.
        
        I.e. expected reduction in entropy of q_node due to finding at 
        finding_node.  Create this Sensitivity object with:  
        q_node.new_sensitivity (EntropyMeasure, ..).  
        Like Netica-C GetMutualInfo_bn.
        """
        if not isinstance(finding_node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(finding_node).__name__))
        Netica.GetMutualInfo_bn.restype = ctypes.c_double
        mutual_info = Netica.GetMutualInfo_bn(ctypes.c_void_p(self.cptr), 
                                              ctypes.c_void_p(finding_node.cptr))
        checkerr()
        return mutual_info  

    def get_varience_of_real(self, finding_node):
        """The expected change squared in the expected real value of query_node, 
        if a finding was obtained for finding_node.  
        
        Create this Sensitivity object with:  query_node.NewSensitivity (
        RealMeasure+VarianceMeasure, ..).  
        Like Netica-C GetVarianceOfReal_bn.
        """
        if not isinstance(finding_node, Node):
            raise TypeError('A Node is required (got type {})'.format(type(finding_node).__name__))
        Netica.GetVarianceOfReal_bn.restype = ctypes.c_double
        varience_of_real = Netica.GetVarianceOfReal_bn(ctypes.c_void_p(self.cptr),
                                                       ctypes.c_void_p(finding_node.cptr))
        checkerr()
        return varience_of_real  






